--[[--------------------------------------------------------------------
	LibRealmInfo
	World of Warcraft library for obtaining information about realms.
	Copyright 2014-2019 Phanx <addons@phanx.net>
	Zlib license. Standalone distribution strongly discouraged.
	https://github.com/phanx-wow/LibRealmInfo
	https://wow.curseforge.com/projects/librealminfo
	https://www.wowinterface.com/downloads/info22987-LibRealmInfo
----------------------------------------------------------------------]]

local MAJOR, MINOR = "LibRealmInfo", 17
assert(LibStub, MAJOR.." requires LibStub")
local lib, oldminor = LibStub:NewLibrary(MAJOR, MINOR)
if not lib then return end

local standalone = (...) == MAJOR
local realmData, connectionData
local Unpack

local function debug(...)
	if standalone then
		print("|cffff7f7f["..MAJOR.."]|r", ...)
	end
end

local function shallowCopy(t)
	if not t then return end

	local n = {}
	for k, v in next, t do
		n[k] = v
	end
	return n
end

local function getNameForAPI(name)
	return name and (name:gsub("[%s%-]", "")) or nil
end

------------------------------------------------------------------------

local currentRegion

function lib:GetCurrentRegion()
	if currentRegion then
		return currentRegion
	end

	if Unpack then
		Unpack()
	end

	local guid = UnitGUID("player")
	if guid then
		local server = tonumber(strmatch(guid, "^Player%-(%d+)"))
		local realm = realmData[server]
		if realm then
			currentRegion = realm.region
			return currentRegion
		end
	end

	debug("GetCurrentRegion: could not identify region based on player GUID", guid)
end

------------------------------------------------------------------------

local validRegions = { US = true, EU = true, CN = true, KR = true, TW = true }

function lib:GetRealmInfo(name, region)
	debug("GetRealmInfo", name, region)
	local isString = type(name) == "string"
	if isString then
		name = strtrim(name)
	end
	if type(name) == "number" or isString and strmatch(name, "^%d+$") then
		return self:GetRealmInfoByID(name)
	end
	assert(isString and strlen(name) > 0, "Usage: GetRealmInfo(name[, region])")

	if not region or not validRegions[region] then
		region = self:GetCurrentRegion()
	end

	if Unpack then
		Unpack()
	end

	for id, realm in pairs(realmData) do
		if realm.region == region and (realm.nameForAPI == name or realm.name == name or realm.englishNameForAPI == name or realm.englishName == name) then
			return id, realm.name, realm.nameForAPI, realm.rules, realm.locale, nil, realm.region, realm.timezone, shallowCopy(realm.connections), realm.englishName, realm.englishNameForAPI
		end
	end

	debug("No info found for realm", name, "in region", region)
end

------------------------------------------------------------------------

function lib:GetRealmInfoByID(id)
	debug("GetRealmInfoByID", id)
	id = tonumber(id)
	assert(id, "Usage: GetRealmInfoByID(id)")

	if Unpack then
		Unpack()
	end

	local realm = realmData[id]
	if realm and realm.name then
		return realm.id, realm.name, realm.nameForAPI, realm.rules, realm.locale, nil, realm.region, realm.timezone, shallowCopy(realm.connections), realm.englishName, realm.englishNameForAPI
	end

	debug("No info found for realm ID", name)
end

------------------------------------------------------------------------

function lib:GetRealmInfoByGUID(guid)
	assert(type(guid) == "string", "Usage: GetRealmInfoByGUID(guid)")
	if not strmatch(guid, "^Player%-") then
		return debug("Unsupported GUID type", (strsplit("-", guid)))
	end
	local _, _, _, _, _, _, realm = GetPlayerInfoByGUID(guid)
	if realm == "" then
		realm = GetRealmName()
	end
	return self:GetRealmInfo(realm)
end

------------------------------------------------------------------------

function lib:GetRealmInfoByUnit(unit)
	assert(type(unit) == "string", "Usage: GetRealmInfoByUnit(unit)")
	local guid = UnitGUID(unit)
	if not guid then
		return debug("No GUID available for unit", unit)
	end
	return self:GetRealmInfoByGUID(guid)
end

------------------------------------------------------------------------

function Unpack()
	debug("Unpacking data...")

	for id, info in pairs(realmData) do
		-- Aegwynn,PvE,enUS,US,CST
		-- Nathrezim,PvE,deDE,EU
		-- Азурегос,PvE,ruRU,EU,Azuregos
		local name, rules, locale, region, timezone = strsplit(",", info)

		local englishName
		if region ~= "US" then
			englishName = timezone
			timezone = nil
		end

		realmData[id] = {
			id = id,
			name = name,
			nameForAPI = getNameForAPI(name),
			rules = string.upper(rules),
			locale = locale,
			region = region,
			timezone = timezone, -- only for realms in US region
			englishName = englishName, -- only for realms with non-Latin names
			englishNameForAPI = getNameForAPI(englishName), -- only for realms with non-Latin names
		}
	end

	for i = 1, #connectionData do
		local connectedRealms = { strsplit(",", connectionData[i]) }
		local connectionID = tonumber(table.remove(connectedRealms, 1))
		local region = table.remove(connectedRealms, 1)

		if not realmData[connectionID] then
			-- nameless server used to host connected realms
			table.insert(connectedRealms, connectionID)
			realmData[connectionID] = {
				region = region,
				connections = connectedRealms
			}
		end

		for j = 1, #connectedRealms do
			local realmID = tonumber(connectedRealms[j])
			connectedRealms[j] = realmID
			realmData[realmID].connections = connectedRealms
		end
	end

	-- Partial workaround for missing Chinese connected realm data:
	local autoCompleteRealms = GetAutoCompleteRealms()
	if #autoCompleteRealms > 0 then
		local autoCompleteIDs = {}
		for _, name in pairs(autoCompleteRealms) do
			for realmID, realm in pairs(realmData) do
				if realm.nameForAPI == name then
					table.insert(autoCompleteIDs, realmID)
					break
				end
			end
		end
		if #autoCompleteIDs == #autoCompleteRealms then
			for _, realmID in pairs(autoCompleteIDs) do
				local realm = realmData[realmID]
				if realm and not realm.connections then
					realm.connections = autoCompleteIDs
				end
			end
		else
			debug("Failed to match names from GetAutoCompleteRealms!")
		end
	end

	connectionData = nil
	Unpack = nil
	collectgarbage()

	debug("Done unpacking data.")
end

------------------------------------------------------------------------

realmData = {
[1]="Lightbringer,PvE,enUS,US,PST",
[2]="Cenarius,PvE,enUS,US,PST",
[3]="Uther,PvE,enUS,US,CST",
[4]="Kilrogg,PvE,enUS,US,PST",
[5]="Proudmoore,PvE,enUS,US,PST",
[6]="Hyjal,PvE,enUS,US,PST",
[7]="Frostwolf,PvE,enUS,US,EST",
[8]="Ner'zhul,PvE,enUS,US,PST",
[9]="Kil'jaeden,PvE,enUS,US,PST",
[10]="Blackrock,PvE,enUS,US,MST",
[11]="Tichondrius,PvE,enUS,US,PST",
[12]="Silver Hand,RP,enUS,US,PST",
[13]="Doomhammer,PvE,enUS,US,PST",
[14]="Icecrown,PvE,enUS,US,CST",
[15]="Deathwing,PvE,enUS,US,PST",
[16]="Kel'Thuzad,PvE,enUS,US,MST",
[47]="Eitrigg,PvE,enUS,US,CST",
[51]="Garona,PvE,enUS,US,CST",
[52]="Alleria,PvE,enUS,US,CST",
[53]="Hellscream,PvE,enUS,US,CST",
[54]="Blackhand,PvE,enUS,US,CST",
[55]="Whisperwind,PvE,enUS,US,CST",
[56]="Archimonde,PvE,enUS,US,CST",
[57]="Illidan,PvE,enUS,US,CST",
[58]="Stormreaver,PvE,enUS,US,CST",
[59]="Mal'Ganis,PvE,enUS,US,CST",
[60]="Stormrage,PvE,enUS,US,EST",
[61]="Zul'jin,PvE,enUS,US,EST",
[62]="Medivh,PvE,enUS,US,CST",
[63]="Durotan,PvE,enUS,US,EST",
[64]="Bloodhoof,PvE,enUS,US,EST",
[65]="Khadgar,PvE,enUS,US,CST",
[66]="Dalaran,PvE,enUS,US,EST",
[67]="Elune,PvE,enUS,US,EST",
[68]="Lothar,PvE,enUS,US,EST",
[69]="Arthas,PvE,enUS,US,EST",
[70]="Mannoroth,PvE,enUS,US,CST",
[71]="Warsong,PvE,enUS,US,EST",
[72]="Shattered Hand,PvE,enUS,US,PST",
[73]="Bleeding Hollow,PvE,enUS,US,EST",
[74]="Skullcrusher,PvE,enUS,US,EST",
[75]="Argent Dawn,RP,enUS,US,EST",
[76]="Sargeras,PvE,enUS,US,CST",
[77]="Azgalor,PvE,enUS,US,CST",
[78]="Magtheridon,PvE,enUS,US,EST",
[79]="Destromath,PvE,enUS,US,CST",
[80]="Gorgonnash,PvE,enUS,US,EST",
[81]="Dethecus,PvE,enUS,US,CST",
[82]="Spinebreaker,PvE,enUS,US,CST",
[83]="Bonechewer,PvE,enUS,US,CST",
[84]="Dragonmaw,PvE,enUS,US,PST",
[85]="Shadowsong,PvE,enUS,US,PST",
[86]="Silvermoon,PvE,enUS,US,PST",
[87]="Windrunner,PvE,enUS,US,PST",
[88]="Cenarion Circle,RP,enUS,US,MST",
[89]="Nathrezim,PvE,enUS,US,CST",
[90]="Terenas,PvE,enUS,US,PST",
[91]="Burning Blade,PvE,enUS,US,CST",
[92]="Gorefiend,PvE,enUS,US,CST",
[93]="Eredar,PvE,enUS,US,CST",
[94]="Shadowmoon,PvE,enUS,US,CST",
[95]="Lightning's Blade,PvE,enUS,US,CST",
[96]="Eonar,PvE,enUS,US,EST",
[97]="Gilneas,PvE,enUS,US,EST",
[98]="Kargath,PvE,enUS,US,CST",
[99]="Llane,PvE,enUS,US,EST",
[100]="Earthen Ring,RP,enUS,US,EST",
[101]="Laughing Skull,PvE,enUS,US,EST",
[102]="Burning Legion,PvE,enUS,US,CST",
[103]="Thunderlord,PvE,enUS,US,CST",
[104]="Malygos,PvE,enUS,US,CST",
[105]="Thunderhorn,PvE,enUS,US,CST",
[106]="Aggramar,PvE,enUS,US,CST",
[107]="Crushridge,PvE,enUS,US,CST",
[108]="Stonemaul,PvE,enUS,US,CST",
[109]="Daggerspine,PvE,enUS,US,CST",
[110]="Stormscale,PvE,enUS,US,EST",
[111]="Dunemaul,PvE,enUS,US,CST",
[112]="Boulderfist,PvE,enUS,US,CST",
[113]="Suramar,PvE,enUS,US,PST",
[114]="Dragonblight,PvE,enUS,US,PST",
[115]="Draenor,PvE,enUS,US,PST",
[116]="Uldum,PvE,enUS,US,PST",
[117]="Bronzebeard,PvE,enUS,US,PST",
[118]="Feathermoon,RP,enUS,US,PST",
[119]="Bloodscalp,PvE,enUS,US,CST",
[120]="Darkspear,PvE,enUS,US,MST",
[121]="Azjol-Nerub,PvE,enUS,US,MST",
[122]="Perenolde,PvE,enUS,US,PST",
[123]="Eldre'Thalas,PvE,enUS,US,PST",
[124]="Spirestone,PvE,enUS,US,EST",
[125]="Shadow Council,RP,enUS,US,MST",
[126]="Scarlet Crusade,RP,enUS,US,PST",
[127]="Firetree,PvE,enUS,US,EST",
[128]="Frostmane,PvE,enUS,US,PST",
[129]="Gurubashi,PvE,enUS,US,CST",
[130]="Smolderthorn,PvE,enUS,US,CST",
[131]="Skywall,PvE,enUS,US,PST",
[151]="Runetotem,PvE,enUS,US,CST",
[153]="Moonrunner,PvE,enUS,US,EST",
[154]="Detheroc,PvE,enUS,US,CST",
[155]="Kalecgos,PvE,enUS,US,PST",
[156]="Ursin,PvE,enUS,US,EST",
[157]="Dark Iron,PvE,enUS,US,PST",
[158]="Greymane,PvE,enUS,US,CST",
[159]="Wildhammer,PvE,enUS,US,CST",
[160]="Staghelm,PvE,enUS,US,CST",
[162]="Emerald Dream,RP,enUS,US,CST",
[163]="Maelstrom,RP,enUS,US,CST",
[164]="Twisting Nether,RP,enUS,US,CST",
[201]="불타는 군단,PvE,koKR,KR,Burning Legion",
[205]="아즈샤라,PvE,koKR,KR,Azshara",
[207]="달라란,PvE,koKR,KR,Dalaran",
[210]="듀로탄,PvE,koKR,KR,Durotan",
[211]="노르간논,PvE,koKR,KR,Norgannon",
[212]="가로나,PvE,koKR,KR,Garona",
[214]="윈드러너,PvE,koKR,KR,Windrunner",
[215]="굴단,PvE,koKR,KR,Gul'dan",
[258]="알렉스트라자,PvE,koKR,KR,Alexstrasza",
[264]="말퓨리온,PvE,koKR,KR,Malfurion",
[293]="헬스크림,PvE,koKR,KR,Hellscream",
[500]="Aggramar,PvE,enUS,EU",
[501]="Arathor,PvE,enUS,EU",
[502]="Aszune,PvE,enUS,EU",
[503]="Azjol-Nerub,PvE,enUS,EU",
[504]="Bloodhoof,PvE,enUS,EU",
[505]="Doomhammer,PvE,enUS,EU",
[506]="Draenor,PvE,enUS,EU",
[507]="Dragonblight,PvE,enUS,EU",
[508]="Emerald Dream,PvE,enUS,EU",
[509]="Garona,PvE,frFR,EU",
[510]="Vol'jin,PvE,frFR,EU",
[511]="Sunstrider,PvE,enUS,EU",
[512]="Arak-arahm,PvE,frFR,EU",
[513]="Twilight's Hammer,PvE,enUS,EU",
[515]="Zenedar,PvE,enUS,EU",
[516]="Forscherliga,RP,deDE,EU",
[517]="Medivh,PvE,frFR,EU",
[518]="Agamaggan,PvE,enUS,EU",
[519]="Al'Akir,PvE,enUS,EU",
[521]="Bladefist,PvE,enUS,EU",
[522]="Bloodscalp,PvE,enUS,EU",
[523]="Burning Blade,PvE,enUS,EU",
[524]="Burning Legion,PvE,enUS,EU",
[525]="Crushridge,PvE,enUS,EU",
[526]="Daggerspine,PvE,enUS,EU",
[527]="Deathwing,PvE,enUS,EU",
[528]="Dragonmaw,PvE,enUS,EU",
[529]="Dunemaul,PvE,enUS,EU",
[531]="Dethecus,PvE,deDE,EU",
[533]="Sinstralis,PvE,frFR,EU",
[535]="Durotan,PvE,deDE,EU",
[536]="Argent Dawn,RP,enUS,EU",
[537]="Kirin Tor,RP,frFR,EU",
[538]="Dalaran,PvE,frFR,EU",
[539]="Archimonde,PvE,frFR,EU",
[540]="Elune,PvE,frFR,EU",
[541]="Illidan,PvE,frFR,EU",
[542]="Hyjal,PvE,frFR,EU",
[543]="Kael'thas,PvE,frFR,EU",
[544]="Ner’zhul,PvE,frFR,EU",
[545]="Cho’gall,PvE,frFR,EU",
[546]="Sargeras,PvE,frFR,EU",
[547]="Runetotem,PvE,enUS,EU",
[548]="Shadowsong,PvE,enUS,EU",
[549]="Silvermoon,PvE,enUS,EU",
[550]="Stormrage,PvE,enUS,EU",
[551]="Terenas,PvE,enUS,EU",
[552]="Thunderhorn,PvE,enUS,EU",
[553]="Turalyon,PvE,enUS,EU",
[554]="Ravencrest,PvE,enUS,EU",
[556]="Shattered Hand,PvE,enUS,EU",
[557]="Skullcrusher,PvE,enUS,EU",
[558]="Spinebreaker,PvE,enUS,EU",
[559]="Stormreaver,PvE,enUS,EU",
[560]="Stormscale,PvE,enUS,EU",
[561]="Earthen Ring,RP,enUS,EU",
[562]="Alexstrasza,PvE,deDE,EU",
[563]="Alleria,PvE,deDE,EU",
[564]="Antonidas,PvE,deDE,EU",
[565]="Baelgun,PvE,deDE,EU",
[566]="Blackhand,PvE,deDE,EU",
[567]="Gilneas,PvE,deDE,EU",
[568]="Kargath,PvE,deDE,EU",
[569]="Khaz'goroth,PvE,deDE,EU",
[570]="Lothar,PvE,deDE,EU",
[571]="Madmortem,PvE,deDE,EU",
[572]="Malfurion,PvE,deDE,EU",
[573]="Zuluhed,PvE,deDE,EU",
[574]="Nozdormu,PvE,deDE,EU",
[575]="Perenolde,PvE,deDE,EU",
[576]="Die Silberne Hand,RP,deDE,EU",
[577]="Aegwynn,PvE,deDE,EU",
[578]="Arthas,PvE,deDE,EU",
[579]="Azshara,PvE,deDE,EU",
[580]="Blackmoore,PvE,deDE,EU",
[581]="Blackrock,PvE,deDE,EU",
[582]="Destromath,PvE,deDE,EU",
[583]="Eredar,PvE,deDE,EU",
[584]="Frostmourne,PvE,deDE,EU",
[585]="Frostwolf,PvE,deDE,EU",
[586]="Gorgonnash,PvE,deDE,EU",
[587]="Gul'dan,PvE,deDE,EU",
[588]="Kel'Thuzad,PvE,deDE,EU",
[589]="Kil'jaeden,PvE,deDE,EU",
[590]="Mal'Ganis,PvE,deDE,EU",
[591]="Mannoroth,PvE,deDE,EU",
[592]="Zirkel des Cenarius,RP,deDE,EU",
[593]="Proudmoore,PvE,deDE,EU",
[594]="Nathrezim,PvE,deDE,EU",
[600]="Dun Morogh,PvE,deDE,EU",
[601]="Aman'thul,PvE,deDE,EU",
[602]="Sen'jin,PvE,deDE,EU",
[604]="Thrall,PvE,deDE,EU",
[605]="Theradras,PvE,deDE,EU",
[606]="Genjuros,PvE,enUS,EU",
[607]="Balnazzar,PvE,enUS,EU",
[608]="Anub'arak,PvE,deDE,EU",
[609]="Wrathbringer,PvE,deDE,EU",
[610]="Onyxia,PvE,deDE,EU",
[611]="Nera'thor,PvE,deDE,EU",
[612]="Nefarian,PvE,deDE,EU",
[613]="Kult der Verdammten,RP,deDE,EU",
[614]="Das Syndikat,RP,deDE,EU",
[615]="Terrordar,PvE,deDE,EU",
[616]="Krag'jin,PvE,deDE,EU",
[617]="Der Rat von Dalaran,RP,deDE,EU",
[618]="Nordrassil,PvE,enUS,EU",
[619]="Hellscream,PvE,enUS,EU",
[621]="Laughing Skull,PvE,enUS,EU",
[622]="Magtheridon,PvE,enUS,EU",
[623]="Quel'Thalas,PvE,enUS,EU",
[624]="Neptulon,PvE,enUS,EU",
[625]="Twisting Nether,PvE,enUS,EU",
[626]="Ragnaros,PvE,enUS,EU",
[627]="The Maelstrom,PvE,enUS,EU",
[628]="Sylvanas,PvE,enUS,EU",
[629]="Vashj,PvE,enUS,EU",
[630]="Bloodfeather,PvE,enUS,EU",
[631]="Darksorrow,PvE,enUS,EU",
[632]="Frostwhisper,PvE,enUS,EU",
[633]="Kor'gall,PvE,enUS,EU",
[635]="Defias Brotherhood,RP,enUS,EU",
[636]="The Venture Co,RP,enUS,EU",
[637]="Lightning's Blade,PvE,enUS,EU",
[638]="Haomarush,PvE,enUS,EU",
[639]="Xavius,PvE,enUS,EU",
[640]="Khaz Modan,PvE,frFR,EU",
[641]="Drek'Thar,PvE,frFR,EU",
[642]="Rashgarroth,PvE,frFR,EU",
[643]="Throk'Feroth,PvE,frFR,EU",
[644]="Conseil des Ombres,RP,frFR,EU",
[645]="Varimathras,PvE,frFR,EU",
[646]="Hakkar,PvE,enUS,EU",
[647]="Les Sentinelles,RP,frFR,EU",
[700]="阿格拉玛,PvE,zhCN,CN,Aggramar",
[703]="艾苏恩,PvE,zhCN,CN,Aszune",
[704]="安威玛尔,PvE,zhCN,CN,Anvilmar",
[705]="奥达曼,PvE,zhCN,CN,Uldaman",
[706]="奥蕾莉亚,PvE,zhCN,CN,Alleria",
[707]="白银之手,PvE,zhCN,CN,Silver Hand",
[708]="暴风祭坛,PvE,zhCN,CN,Alter of Storms",
[709]="藏宝海湾,PvE,zhCN,CN,Booty Bay",
[710]="尘风峡谷,PvE,zhCN,CN,Dustwind Gulch",
[711]="达纳斯,PvE,zhCN,CN,Danath Trollbane",
[712]="迪托马斯,PvE,zhCN,CN,Destromath",
[714]="国王之谷,PvE,zhCN,CN,Valley of Kings",
[715]="黑龙军团,PvE,zhCN,CN,Black Dragonflight",
[716]="黑石尖塔,PvE,zhCN,CN,Blackrock",
[717]="红龙军团,PvE,zhCN,CN,Red Dragonflight",
[718]="回音山,PvE,zhCN,CN,Echo Ridge",
[719]="基尔罗格,PvE,zhCN,CN,Kilrogg",
[720]="卡德罗斯,PvE,zhCN,CN,Khardros",
[721]="卡扎克,PvE,zhCN,CN,Lord Kazzak",
[723]="库德兰,PvE,zhCN,CN,Kurdran",
[725]="蓝龙军团,PvE,zhCN,CN,Blue Dragonflight",
[726]="雷霆之王,PvE,zhCN,CN,Thunderlord",
[727]="烈焰峰,PvE,zhCN,CN,Flame Crest",
[729]="罗宁,PvE,zhCN,CN,Rhonin",
[730]="洛萨,PvE,zhCN,CN,Lothar",
[731]="玛多兰,PvE,zhCN,CN,Madoran",
[732]="玛瑟里顿,PvE,zhCN,CN,Magtheridon",
[734]="奈萨里奥,PvE,zhCN,CN,Neltharion",
[736]="诺莫瑞根,PvE,zhCN,CN,Gnomeregan",
[737]="普瑞斯托,PvE,zhCN,CN,Prestor",
[738]="燃烧平原,PvE,zhCN,CN,Burning Steppes",
[739]="萨格拉斯,PvE,zhCN,CN,Sargeras",
[740]="山丘之王,PvE,zhCN,CN,Bronzebeard",
[741]="死亡之翼,PvE,zhCN,CN,Deathwing",
[742]="索拉丁,PvE,zhCN,CN,Thoradin",
[743]="索瑞森,PvE,zhCN,CN,Thaurissan",
[744]="铜龙军团,PvE,zhCN,CN,Bronze Dragonflight",
[745]="图拉扬,PvE,zhCN,CN,Turalyon",
[746]="伊瑟拉,PvE,zhCN,CN,Ysera",
[748]="阿迦玛甘,PvE,zhCN,CN,Agamaggan",
[749]="阿克蒙德,PvE,zhCN,CN,Archimonde",
[750]="埃加洛尔,PvE,zhCN,CN,Azgalor",
[751]="埃苏雷格,PvE,zhCN,CN,Azuregos",
[753]="艾萨拉,PvE,zhCN,CN,Azshara",
[754]="艾森娜,PvE,zhCN,CN,Aessina",
[755]="爱斯特纳,PvE,zhCN,CN,Astranaar",
[756]="暗影之月,PvE,zhCN,CN,Shadowmoon",
[757]="奥拉基尔,PvE,zhCN,CN,Al'Akir",
[758]="冰霜之刃,PvE,zhCN,CN,Frostmane",
[760]="达斯雷玛,PvE,zhCN,CN,Dath'Remar",
[761]="地狱咆哮,PvE,zhCN,CN,Hellscream",
[762]="地狱之石,PvE,zhCN,CN,Fel Rock",
[764]="风暴之怒,PvE,zhCN,CN,Stormrage",
[765]="风行者,PvE,zhCN,CN,Windrunner",
[766]="弗塞雷迦,PvE,zhCN,CN,Explorer's League",
[767]="戈古纳斯,PvE,zhCN,CN,Gorgonnash",
[768]="海加尔,PvE,zhCN,CN,Hyjal",
[769]="毁灭之锤,PvE,zhCN,CN,Doomhammer",
[770]="火焰之树,PvE,zhCN,CN,Firetree",
[771]="卡德加,PvE,zhCN,CN,Khadgar",
[772]="拉文凯斯,PvE,zhCN,CN,Ravencrest",
[773]="玛法里奥,PvE,zhCN,CN,Malfurion",
[774]="玛维·影歌,PvE,zhCN,CN,Maiev Shadowsong",
[775]="梅尔加尼,PvE,zhCN,CN,Mal'Ganis",
[776]="梦境之树,PvE,zhCN,CN,Dream Bough",
[778]="耐普图隆,PvE,zhCN,CN,Neptulon",
[780]="轻风之语,PvE,zhCN,CN,Whisperwind",
[781]="夏维安,PvE,zhCN,CN,Xavian",
[782]="塞纳留斯,PvE,zhCN,CN,Cenarius",
[784]="闪电之刃,PvE,zhCN,CN,Lightning's Blade",
[786]="石爪峰,PvE,zhCN,CN,Stonetalon Peak",
[787]="泰兰德,PvE,zhCN,CN,Tyrande",
[788]="屠魔山谷,PvE,zhCN,CN,Demon Fall Canyon",
[790]="伊利丹,PvE,zhCN,CN,Illidan",
[791]="月光林地,PvE,zhCN,CN,Moonglade",
[792]="月神殿,PvE,zhCN,CN,Temple of Elune",
[793]="战歌,PvE,zhCN,CN,Warsong",
[794]="主宰之剑,PvE,zhCN,CN,The Master's Glaive",
[797]="埃德萨拉,PvE,zhCN,CN,Eldre'Thalas",
[799]="血环,PvE,zhCN,CN,Bleeding Hollow",
[800]="布莱克摩,PvE,zhCN,CN,Blackmoore",
[802]="杜隆坦,PvE,zhCN,CN,Durotan",
[803]="符文图腾,PvE,zhCN,CN,Runetotem",
[804]="鬼雾峰,PvE,zhCN,CN,Dreadmist Peak",
[805]="黑暗之矛,PvE,zhCN,CN,Darkspear",
[806]="红龙女王,PvE,zhCN,CN,Alexstrasza",
[807]="红云台地,PvE,zhCN,CN,Red Cloud Mesa",
[808]="黄金之路,PvE,zhCN,CN,Gold Road",
[810]="火羽山,PvE,zhCN,CN,Fire Plume Ridge",
[812]="迦罗娜,PvE,zhCN,CN,Garona",
[814]="凯恩血蹄,PvE,zhCN,CN,Bloodhoof",
[815]="狂风峭壁,PvE,zhCN,CN,Windshear Crag",
[816]="雷斧堡垒,PvE,zhCN,CN,Thunder Axe Fortress",
[817]="雷克萨,PvE,zhCN,CN,Rexxar",
[818]="雷霆号角,PvE,zhCN,CN,Thunderhorn",
[822]="玛里苟斯,PvE,zhCN,CN,Malygos",
[825]="纳沙塔尔,PvE,zhCN,CN,Nazjatar",
[826]="诺兹多姆,PvE,zhCN,CN,Nozdormu",
[827]="普罗德摩,PvE,zhCN,CN,Proudmoore",
[828]="千针石林,PvE,zhCN,CN,Thousand Needles",
[829]="燃烧之刃,PvE,zhCN,CN,Burning Blade",
[830]="萨尔,PvE,zhCN,CN,Thrall",
[833]="圣火神殿,PvE,zhCN,CN,Shrine of the Dormant Flame",
[835]="甜水绿洲,PvE,zhCN,CN,Lushwater Oasis",
[837]="沃金,PvE,zhCN,CN,Vol'jin",
[838]="熊猫酒仙,PvE,zhCN,CN,Pandaren",
[839]="血牙魔王,PvE,zhCN,CN,Gorefiend",
[840]="勇士岛,PvE,zhCN,CN,Fray Island",
[841]="羽月,PvE,zhCN,CN,Feathermoon",
[842]="蜘蛛王国,PvE,zhCN,CN,Azjol-Nerub",
[843]="自由之风,PvE,zhCN,CN,Freewind",
[844]="阿尔萨斯,PvE,zhCN,CN,Arthas",
[845]="阿拉索,PvE,zhCN,CN,Arathor",
[846]="埃雷达尔,PvE,zhCN,CN,Eredar",
[847]="艾欧纳尔,PvE,zhCN,CN,Eonar",
[848]="安东尼达斯,PvE,zhCN,CN,Antonidas",
[849]="暗影议会,PvE,zhCN,CN,Shadow Council",
[850]="奥特兰克,PvE,zhCN,CN,Alterac Mountains",
[851]="巴尔古恩,PvE,zhCN,CN,Baelgun",
[852]="冰风岗,PvE,zhCN,CN,Chillwind Point",
[855]="达隆米尔,PvE,zhCN,CN,Darrowmere",
[856]="耳语海岸,PvE,zhCN,CN,Whispering Shore",
[857]="古尔丹,PvE,zhCN,CN,Gul'dan",
[858]="寒冰皇冠,PvE,zhCN,CN,Icecrown",
[859]="基尔加丹,PvE,zhCN,CN,Kil'Jaeden",
[860]="激流堡,PvE,zhCN,CN,Stromgarde Keep",
[861]="巨龙之吼,PvE,zhCN,CN,Dragonmaw",
[862]="暗影裂口,PvE,zhCN,CN,Cleft of Shadow",
[863]="凯尔萨斯,PvE,zhCN,CN,Kael'thas",
[864]="克尔苏加德,PvE,zhCN,CN,Kel'Thuzad",
[865]="拉格纳洛斯,PvE,zhCN,CN,Ragnaros",
[866]="埃霍恩,PvE,zhCN,CN,Ahune",
[867]="利刃之拳,PvE,zhCN,CN,Bladefist",
[869]="玛诺洛斯,PvE,zhCN,CN,Mannoroth",
[870]="麦迪文,PvE,zhCN,CN,Medivh",
[872]="耐奥祖,PvE,zhCN,CN,Ner'zhul",
[874]="瑞文戴尔,PvE,zhCN,CN,Rivendare",
[876]="霜狼,PvE,zhCN,CN,Frostwolf",
[877]="霜之哀伤,PvE,zhCN,CN,Frostmourne",
[878]="斯坦索姆,PvE,zhCN,CN,Stratholme",
[879]="塔伦米尔,PvE,zhCN,CN,Tarren Mill",
[882]="提瑞斯法,PvE,zhCN,CN,Tirisfal Glades",
[883]="通灵学院,PvE,zhCN,CN,Scholomance",
[885]="希尔瓦娜斯,PvE,zhCN,CN,Sylvanas",
[886]="血色十字军,PvE,zhCN,CN,Scarlet Crusade",
[887]="遗忘海岸,PvE,zhCN,CN,The Forgotten Coast",
[888]="银松森林,PvE,zhCN,CN,Silverpine Forest",
[889]="银月,PvE,zhCN,CN,Silvermoon",
[890]="鹰巢山,PvE,zhCN,CN,Aerie Peak",
[891]="影牙要塞,PvE,zhCN,CN,Shadowfang Keep",
[915]="狂热之刃,PvE,zhCN,CN,Zealot Blade",
[916]="卡珊德拉,PvE,zhCN,CN,Cassandra",
[917]="迅捷微风,PvE,zhCN,CN,Swiftwind",
[918]="守护之剑,PvE,zhCN,CN,Guardian Blade",
[920]="斩魔者,PvE,zhCN,CN,Demonslayer",
[921]="布兰卡德,PvE,zhCN,CN,Blanchard",
[922]="世界之树,PvE,zhCN,CN,World Tree",
[924]="恶魔之翼,PvE,zhCN,CN,Direwing",
[925]="万色星辰,PvE,zhCN,CN,Keleidoscope Star",
[926]="激流之傲,PvE,zhCN,CN,River Pride",
[927]="加兹鲁维,PvE,zhCN,CN,Gazlowe",
[929]="苏塔恩,PvE,zhCN,CN,Sutarn",
[930]="大地之怒,PvE,zhCN,CN,Deepfury",
[931]="雏龙之翼,PvE,zhCN,CN,Wing of the Whelping",
[932]="黑暗魅影,PvE,zhCN,CN,Dark Phantom",
[933]="踏梦者,PvE,zhCN,CN,Dreamwalker",
[938]="密林游侠,PvE,zhCN,CN,Rangers",
[940]="伊森利恩,PvE,zhCN,CN,Isillien",
[941]="神圣之歌,PvE,zhCN,CN,Holy Chanter",
[943]="暮色森林,PvE,zhCN,CN,Duskwood",
[944]="元素之力,PvE,zhCN,CN,Force of Elemental",
[946]="日落沼泽,PvE,zhCN,CN,Sundown Marsh",
[949]="芬里斯,PvE,zhCN,CN,Fenris",
[951]="伊萨里奥斯,PvE,zhCN,CN,Itharius",
[953]="风暴之眼,PvE,zhCN,CN,Storm Eye",
[954]="提尔之手,PvE,zhCN,CN,Tyr's Hand",
[956]="永夜港,PvE,zhCN,CN,Nighthaven",
[959]="朵丹尼尔,PvE,zhCN,CN,Dor'Danil",
[960]="法拉希姆,PvE,zhCN,CN,Falathim",
[962]="金色平原,RP,zhCN,CN,The Golden Plains",
[963]="暗影之月,PvE,zhTW,TW,Shadowmoon",
[964]="尖石,PvE,zhTW,TW,Spirestone",
[965]="雷鱗,PvE,zhTW,TW,Stormscale",
[966]="巨龍之喉,PvE,zhTW,TW,Dragonmaw",
[977]="冰霜之刺,PvE,zhTW,TW,Frostmane",
[978]="日落沼澤,PvE,zhTW,TW,Sundown Marsh",
[979]="地獄吼,PvE,zhTW,TW,Hellscream",
[980]="天空之牆,PvE,zhTW,TW,Skywall",
[982]="世界之樹,PvE,zhTW,TW,World Tree",
[985]="水晶之刺,PvE,zhTW,TW,Crystalpine Stinger",
[999]="狂熱之刃,PvE,zhTW,TW,Zealot Blade",
[1001]="冰風崗哨,PvE,zhTW,TW,Chillwind Point",
[1006]="米奈希爾,PvE,zhTW,TW,Menethil",
[1023]="屠魔山谷,PvE,zhTW,TW,Demon Fall Canyon",
[1033]="語風,PvE,zhTW,TW,Whisperwind",
[1037]="血之谷,PvE,zhTW,TW,Bleeding Hollow",
[1038]="亞雷戈斯,PvE,zhTW,TW,Arygos",
[1043]="夜空之歌,PvE,zhTW,TW,Nightsong",
[1046]="聖光之願,PvE,zhTW,TW,Light's Hope",
[1048]="銀翼要塞,PvE,zhTW,TW,Silverwing Hold",
[1049]="憤怒使者,PvE,zhTW,TW,Wrathbringer",
[1054]="阿薩斯,PvE,zhTW,TW,Arthas",
[1056]="眾星之子,PvE,zhTW,TW,Quel'dorei",
[1057]="寒冰皇冠,PvE,zhTW,TW,Icecrown",
[1067]="Cho'gall,PvE,enUS,US,EST",
[1068]="Gul'dan,PvE,enUS,US,EST",
[1069]="Kael'thas,PvE,enUS,US,EST",
[1070]="Alexstrasza,PvE,enUS,US,CST",
[1071]="Kirin Tor,RP,enUS,US,CST",
[1072]="Ravencrest,PvE,enUS,US,CST",
[1075]="Balnazzar,PvE,enUS,US,EST",
[1080]="Khadgar,PvE,enUS,EU",
[1081]="Bronzebeard,PvE,enUS,EU",
[1082]="Kul Tiras,PvE,enUS,EU",
[1083]="Chromaggus,PvE,enUS,EU",
[1084]="Dentarg,PvE,enUS,EU",
[1085]="Moonglade,RP,enUS,EU",
[1086]="La Croisade écarlate,RP,frFR,EU",
[1087]="Executus,PvE,enUS,EU",
[1088]="Trollbane,PvE,enUS,EU",
[1089]="Mazrigos,PvE,enUS,EU",
[1090]="Talnivarr,PvE,enUS,EU",
[1091]="Emeriss,PvE,enUS,EU",
[1092]="Drak'thul,PvE,enUS,EU",
[1093]="Ahn'Qiraj,PvE,enUS,EU",
[1096]="Scarshield Legion,RP,enUS,EU",
[1097]="Ysera,PvE,deDE,EU",
[1098]="Malygos,PvE,deDE,EU",
[1099]="Rexxar,PvE,deDE,EU",
[1104]="Anetheron,PvE,deDE,EU",
[1105]="Nazjatar,PvE,deDE,EU",
[1106]="Tichondrius,PvE,deDE,EU",
[1117]="Steamwheedle Cartel,RP,enUS,EU",
[1118]="Die ewige Wacht,RP,deDE,EU",
[1119]="Die Todeskrallen,RP,deDE,EU",
[1121]="Die Arguswacht,RP,deDE,EU",
[1122]="Uldaman,PvE,frFR,EU",
[1123]="Eitrigg,PvE,frFR,EU",
[1127]="Confrérie du Thorium,RP,frFR,EU",
[1128]="Azshara,PvE,enUS,US,CST",
[1129]="Agamaggan,PvE,enUS,US,CST",
[1130]="Lightninghoof,RP,enUS,US,CST",
[1131]="Nazjatar,PvE,enUS,US,CST",
[1132]="Malfurion,PvE,enUS,US,EST",
[1136]="Aegwynn,PvE,enUS,US,CST",
[1137]="Akama,PvE,enUS,US,PST",
[1138]="Chromaggus,PvE,enUS,US,CST",
[1139]="Draka,PvE,enUS,US,PST",
[1140]="Drak'thul,PvE,enUS,US,PST",
[1141]="Garithos,PvE,enUS,US,CST",
[1142]="Hakkar,PvE,enUS,US,CST",
[1143]="Khaz Modan,PvE,enUS,US,MST",
[1145]="Mug'thol,PvE,enUS,US,PST",
[1146]="Korgath,PvE,enUS,US,PST",
[1147]="Kul Tiras,PvE,enUS,US,CST",
[1148]="Malorne,PvE,enUS,US,EST",
[1151]="Rexxar,PvE,enUS,US,CST",
[1154]="Thorium Brotherhood,RP,enUS,US,PST",
[1165]="Arathor,PvE,enUS,US,CST",
[1173]="Madoran,PvE,enUS,US,CST",
[1175]="Trollbane,PvE,enUS,US,EST",
[1182]="Muradin,PvE,enUS,US,MST",
[1184]="Vek'nilash,PvE,enUS,US,CST",
[1185]="Sen'jin,PvE,enUS,US,CST",
[1190]="Baelgun,PvE,enUS,US,PST",
[1198]="安其拉,PvE,zhCN,CN,Ahn'Qiraj",
[1199]="安纳塞隆,PvE,zhCN,CN,Anetheron",
[1200]="阿努巴拉克,PvE,zhCN,CN,Anub'arak",
[1201]="阿拉希,PvE,zhCN,CN,Arathi",
[1202]="瓦里玛萨斯,PvE,zhCN,CN,Varimathras",
[1203]="巴纳扎尔,PvE,zhCN,CN,Balnazzar",
[1204]="黑手军团,PvE,zhCN,CN,Blackhand",
[1205]="血羽,PvE,zhCN,CN,Bloodfeather",
[1206]="燃烧军团,PvE,zhCN,CN,Burning Legion",
[1207]="克洛玛古斯,PvE,zhCN,CN,Chromaggus",
[1208]="破碎岭,PvE,zhCN,CN,Crushridge",
[1209]="克苏恩,PvE,zhCN,CN,C'Thun",
[1210]="阿纳克洛斯,PvE,zhCN,CN,Anachronos",
[1211]="雷霆之怒,PvE,zhCN,CN,Thunder Bluff",
[1212]="桑德兰,PvE,zhCN,CN,Thunderaan",
[1213]="黑翼之巢,PvE,zhCN,CN,Blackwing Lair",
[1214]="德拉诺,PvE,zhCN,CN,Draenor",
[1215]="龙骨平原,PvE,zhCN,CN,Dragonblight",
[1216]="卡拉赞,PvE,zhCN,CN,Karazhan",
[1221]="熔火之心,PvE,zhCN,CN,Molten Core",
[1222]="格瑞姆巴托,PvE,zhCN,CN,Grim Batol",
[1223]="古拉巴什,PvE,zhCN,CN,Gurubashi",
[1224]="哈卡,PvE,zhCN,CN,Hakkar",
[1225]="海克泰尔,PvE,zhCN,CN,Hectae",
[1226]="库尔提拉斯,PvE,zhCN,CN,Kul Tiras",
[1227]="洛丹伦,PvE,zhCN,CN,Lordaeron",
[1228]="奈法利安,PvE,zhCN,CN,Nefarian",
[1229]="奎尔萨拉斯,PvE,zhCN,CN,Quel'Thalas",
[1230]="拉贾克斯,PvE,zhCN,CN,Rajaxx",
[1231]="拉文霍德,PvE,zhCN,CN,Ravenholdt",
[1232]="森金,PvE,zhCN,CN,Sen'jin",
[1233]="范达尔鹿盔,PvE,zhCN,CN,Staghelm",
[1234]="泰拉尔,PvE,zhCN,CN,Taerar",
[1235]="瓦拉斯塔兹,PvE,zhCN,CN,Vaelastrasz",
[1236]="永恒之井,PvE,zhCN,CN,Well of Eternity",
[1237]="海达希亚,PvE,zhCN,CN,Hydraxis",
[1238]="萨菲隆,PvE,zhCN,CN,Sapphiron",
[1239]="纳克萨玛斯,PvE,zhCN,CN,Naxxramas",
[1240]="无尽之海,PvE,zhCN,CN,The Great Sea",
[1241]="莱索恩,PvE,zhCN,CN,Lethon",
[1258]="Duskwood,PvE,enUS,US,EST",
[1259]="Zuluhed,PvE,enUS,US,EST",
[1260]="Steamwheedle Cartel,RP,enUS,US,CST",
[1262]="Norgannon,PvE,enUS,US,CST",
[1263]="Thrall,PvE,enUS,US,EST",
[1264]="Anetheron,PvE,enUS,US,EST",
[1265]="Turalyon,PvE,enUS,US,EST",
[1266]="Haomarush,PvE,enUS,US,CST",
[1267]="Scilla,PvE,enUS,US,EST",
[1268]="Ysondre,PvE,enUS,US,EST",
[1270]="Ysera,PvE,enUS,US,EST",
[1271]="Dentarg,PvE,enUS,US,CST",
[1276]="Andorhal,PvE,enUS,US,EST",
[1277]="Executus,PvE,enUS,US,PST",
[1278]="Dalvengyr,PvE,enUS,US,PST",
[1280]="Black Dragonflight,PvE,enUS,US,EST",
[1282]="Altar of Storms,PvE,enUS,US,EST",
[1283]="Uldaman,PvE,enUS,US,CST",
[1284]="Aerie Peak,PvE,enUS,US,PST",
[1285]="Onyxia,PvE,enUS,US,CST",
[1286]="Demon Soul,PvE,enUS,US,PST",
[1287]="Gnomeregan,PvE,enUS,US,EST",
[1288]="Anvilmar,PvE,enUS,US,EST",
[1289]="The Venture Co,RP,enUS,US,CST",
[1290]="Sentinels,RP,enUS,US,CST",
[1291]="Jaedenar,PvE,enUS,US,CST",
[1292]="Tanaris,PvE,enUS,US,CST",
[1293]="Alterac Mountains,PvE,enUS,US,EST",
[1294]="Undermine,PvE,enUS,US,EST",
[1295]="Lethon,PvE,enUS,US,CST",
[1296]="Blackwing Lair,PvE,enUS,US,CST",
[1297]="Arygos,PvE,enUS,US,EST",
[1298]="Vek'nilash,PvE,enUS,EU",
[1299]="Boulderfist,PvE,enUS,EU",
[1300]="Frostmane,PvE,enUS,EU",
[1301]="Outland,PvE,enUS,EU",
[1303]="Grim Batol,PvE,enUS,EU",
[1304]="Jaedenar,PvE,enUS,EU",
[1305]="Kazzak,PvE,enUS,EU",
[1306]="Tarren Mill,PvE,enUS,EU",
[1307]="Chamber of Aspects,PvE,enUS,EU",
[1308]="Ravenholdt,RP,enUS,EU",
[1309]="Pozzo dell'Eternità,PvE,itIT,EU",
[1310]="Eonar,PvE,enUS,EU",
[1311]="Kilrogg,PvE,enUS,EU",
[1312]="Aerie Peak,PvE,enUS,EU",
[1313]="Wildhammer,PvE,enUS,EU",
[1314]="Saurfang,PvE,enUS,EU",
[1316]="Nemesis,PvE,itIT,EU",
[1317]="Darkmoon Faire,RP,enUS,EU",
[1318]="Vek'lor,PvE,deDE,EU",
[1319]="Mug'thol,PvE,deDE,EU",
[1320]="Taerar,PvE,deDE,EU",
[1321]="Dalvengyr,PvE,deDE,EU",
[1322]="Rajaxx,PvE,deDE,EU",
[1323]="Ulduar,PvE,deDE,EU",
[1324]="Malorne,PvE,deDE,EU",
[1326]="Der Abyssische Rat,RP,deDE,EU",
[1327]="Der Mithrilorden,RP,deDE,EU",
[1328]="Tirion,PvE,deDE,EU",
[1330]="Ambossar,PvE,deDE,EU",
[1331]="Suramar,PvE,frFR,EU",
[1332]="Krasus,PvE,frFR,EU",
[1333]="Die Nachtwache,RP,deDE,EU",
[1334]="Arathi,PvE,frFR,EU",
[1335]="Ysondre,PvE,frFR,EU",
[1336]="Eldre'Thalas,PvE,frFR,EU",
[1337]="Culte de la Rive noire,RP,frFR,EU",
[1342]="Echo Isles,PvE,enUS,US,PST",
[1344]="The Forgotten Coast,PvE,enUS,US,EST",
[1345]="Fenris,PvE,enUS,US,PST",
[1346]="Anub'arak,PvE,enUS,US,CST",
[1347]="Blackwater Raiders,RP,enUS,US,MST",
[1348]="Vashj,PvE,enUS,US,EST",
[1349]="Korialstrasz,PvE,enUS,US,PST",
[1350]="Misha,PvE,enUS,US,CST",
[1351]="Darrowmere,PvE,enUS,US,PST",
[1352]="Ravenholdt,RP,enUS,US,CST",
[1353]="Bladefist,PvE,enUS,US,CST",
[1354]="Shu'halo,PvE,enUS,US,CST",
[1355]="Winterhoof,PvE,enUS,US,PST",
[1356]="Sisters of Elune,RP,enUS,US,MST",
[1357]="Maiev,PvE,enUS,US,CST",
[1358]="Rivendare,PvE,enUS,US,EST",
[1359]="Nordrassil,PvE,enUS,US,MST",
[1360]="Tortheldrin,PvE,enUS,US,PST",
[1361]="Cairne,PvE,enUS,US,PST",
[1362]="Drak'Tharon,PvE,enUS,US,EST",
[1363]="Antonidas,PvE,enUS,US,PST",
[1364]="Shandris,PvE,enUS,US,PST",
[1365]="Moon Guard,RP,enUS,US,CST",
[1367]="Nazgrel,PvE,enUS,US,CST",
[1368]="Hydraxis,PvE,enUS,US,PST",
[1369]="Wyrmrest Accord,RP,enUS,US,PST",
[1370]="Farstriders,RP,enUS,US,PST",
[1371]="Borean Tundra,PvE,enUS,US,PST",
[1372]="Quel'dorei,PvE,enUS,US,CST",
[1373]="Garrosh,PvE,enUS,US,CST",
[1374]="Mok'Nathal,PvE,enUS,US,PST",
[1375]="Nesingwary,PvE,enUS,US,CST",
[1377]="Drenden,PvE,enUS,US,CST",
[1378]="Dun Modr,PvE,esES,EU",
[1379]="Zul'jin,PvE,esES,EU",
[1380]="Uldum,PvE,esES,EU",
[1381]="C'Thun,PvE,esES,EU",
[1382]="Sanguino,PvE,esES,EU",
[1383]="Shen'dralar,PvE,esES,EU",
[1384]="Tyrande,PvE,esES,EU",
[1385]="Exodar,PvE,esES,EU",
[1386]="Minahonda,PvE,esES,EU",
[1387]="Los Errantes,PvE,esES,EU",
[1388]="Lightbringer,PvE,enUS,EU",
[1389]="Darkspear,PvE,enUS,EU",
[1391]="Alonsus,PvE,enUS,EU",
[1392]="Burning Steppes,PvE,enUS,EU",
[1393]="Bronze Dragonflight,PvE,enUS,EU",
[1394]="Anachronos,PvE,enUS,EU",
[1395]="Colinas Pardas,PvE,esES,EU",
[1400]="Un'Goro,PvE,deDE,EU",
[1401]="Garrosh,PvE,deDE,EU",
[1404]="Area 52,PvE,deDE,EU",
[1405]="Todeswache,RP,deDE,EU",
[1406]="Arygos,PvE,deDE,EU",
[1407]="Teldrassil,PvE,deDE,EU",
[1408]="Norgannon,PvE,deDE,EU",
[1409]="Lordaeron,PvE,deDE,EU",
[1413]="Aggra (Português),PvE,enUS,EU",
[1415]="Terokkar,PvE,enUS,EU",
[1416]="Blade's Edge,PvE,enUS,EU",
[1417]="Azuremyst,PvE,enUS,EU",
[1425]="Drakkari,PvE,esMX,US,CST",
[1427]="Ragnaros,PvE,esMX,US,CST",
[1428]="Quel'Thalas,PvE,esMX,US,CST",
[1482]="阿卡玛,PvE,zhCN,CN,Akama",
[1483]="阿扎达斯,PvE,zhCN,CN,Archaedas",
[1484]="灰谷,PvE,zhCN,CN,Ashenvale",
[1485]="艾维娜,PvE,zhCN,CN,Aviana",
[1486]="巴瑟拉斯,PvE,zhCN,CN,Barthilas",
[1487]="血顶,PvE,zhCN,CN,Bloodscalp",
[1488]="恐怖图腾,PvE,zhCN,CN,Grimtotem",
[1489]="古加尔,PvE,zhCN,CN,Cho'gall",
[1490]="达文格尔,PvE,zhCN,CN,Dalvengyr",
[1491]="黑铁,PvE,zhCN,CN,Dark Iron",
[1492]="恶魔之魂,PvE,zhCN,CN,Demon Soul",
[1493]="迪瑟洛克,PvE,zhCN,CN,Detheroc",
[1494]="丹莫德,PvE,zhCN,CN,Dun Modr",
[1495]="艾莫莉丝,PvE,zhCN,CN,Emeriss",
[1496]="埃克索图斯,PvE,zhCN,CN,Executus",
[1497]="菲拉斯,PvE,zhCN,CN,Feralas",
[1498]="加基森,PvE,zhCN,CN,Gadgetzan",
[1499]="加里索斯,PvE,zhCN,CN,Garithos",
[1500]="格雷迈恩,PvE,zhCN,CN,Greymane",
[1501]="布莱恩,PvE,zhCN,CN,Brann",
[1502]="伊莫塔尔,PvE,zhCN,CN,Immol'thar",
[1503]="大漩涡,PvE,zhCN,CN,Maelstrom",
[1504]="诺森德,PvE,zhCN,CN,Northrend",
[1505]="奥妮克希亚,PvE,zhCN,CN,Onyxia",
[1506]="奥斯里安,PvE,zhCN,CN,Ossirian",
[1507]="外域,PvE,zhCN,CN,Outland",
[1508]="天空之墙,PvE,zhCN,CN,Skywall",
[1509]="风暴之鳞,PvE,zhCN,CN,Stormscale",
[1510]="荆棘谷,PvE,zhCN,CN,Stranglethorn",
[1511]="逐日者,PvE,zhCN,CN,Sunstrider",
[1512]="塔纳利斯,PvE,zhCN,CN,Tanaris",
[1513]="瑟莱德丝,PvE,zhCN,CN,Theradras",
[1514]="塞拉赞恩,PvE,zhCN,CN,Therazane",
[1515]="托塞德林,PvE,zhCN,CN,Tortheldrin",
[1516]="黑暗虚空,PvE,zhCN,CN,Twisting Nether",
[1517]="安戈洛,PvE,zhCN,CN,Un'goro",
[1519]="祖尔金,PvE,zhCN,CN,Zul'jin",
[1520]="双子峰,PvE,zhCN,CN,Twin Peaks",
[1524]="天谴之门,PvE,zhCN,CN,Wrath Gate",
[1549]="Azuremyst,PvE,enUS,US,CST",
[1555]="Auchindoun,PvE,enUS,US,EST",
[1556]="Coilfang,PvE,enUS,US,PST",
[1557]="Shattered Halls,PvE,enUS,US,PST",
[1558]="Blood Furnace,PvE,enUS,US,CST",
[1559]="The Underbog,PvE,enUS,US,CST",
[1563]="Terokkar,PvE,enUS,US,CST",
[1564]="Blade's Edge,PvE,enUS,US,CST",
[1565]="Exodar,PvE,enUS,US,CST",
[1566]="Area 52,PvE,enUS,US,EST",
[1567]="Velen,PvE,enUS,US,EST",
[1570]="The Scryers,RP,enUS,US,EST",
[1572]="Zangarmarsh,PvE,enUS,US,CST",
[1576]="Fizzcrank,PvE,enUS,US,CST",
[1578]="Ghostlands,PvE,enUS,US,EST",
[1579]="Grizzly Hills,PvE,enUS,US,EST",
[1581]="Galakrond,PvE,enUS,US,CST",
[1582]="Dawnbringer,PvE,enUS,US,CST",
[1587]="Hellfire,PvE,enUS,EU",
[1588]="Ghostlands,PvE,enUS,EU",
[1589]="Nagrand,PvE,enUS,EU",
[1595]="The Sha'tar,RP,enUS,EU",
[1596]="Karazhan,PvE,enUS,EU",
[1597]="Auchindoun,PvE,enUS,EU",
[1598]="Shattered Halls,PvE,enUS,EU",
[1602]="Гордунни,PvE,ruRU,EU,Gordunni",
[1603]="Король-лич,PvE,ruRU,EU,Lich King",
[1604]="Свежеватель Душ,PvE,ruRU,EU,Soulflayer",
[1605]="Страж Смерти,PvE,ruRU,EU,Deathguard",
[1606]="Sporeggar,RP,enUS,EU",
[1607]="Nethersturm,PvE,deDE,EU",
[1608]="Shattrath,PvE,deDE,EU",
[1609]="Подземье,PvE,ruRU,EU,Deepholm",
[1610]="Седогрив,PvE,ruRU,EU,Greymane",
[1611]="Festung der Stürme,PvE,deDE,EU",
[1612]="Echsenkessel,PvE,deDE,EU",
[1613]="Blutkessel,PvE,deDE,EU",
[1614]="Галакронд,PvE,ruRU,EU,Galakrond",
[1615]="Ревущий фьорд,PvE,ruRU,EU,Howling Fjord",
[1616]="Разувий,PvE,ruRU,EU,Razuvious",
[1617]="Ткач Смерти,PvE,ruRU,EU,Deathweaver",
[1618]="Die Aldor,RP,deDE,EU",
[1619]="Das Konsortium,RP,deDE,EU",
[1620]="Chants éternels,PvE,frFR,EU",
[1621]="Marécage de Zangar,PvE,frFR,EU",
[1622]="Temple noir,PvE,frFR,EU",
[1623]="Дракономор,PvE,ruRU,EU,Fordragon",
[1624]="Naxxramas,PvE,frFR,EU",
[1625]="Борейская тундра,PvE,ruRU,EU,Borean Tundra",
[1626]="Les Clairvoyants,RP,frFR,EU",
[1657]="冰川之拳,PvE,zhCN,CN,Boulderfist",
[1658]="刺骨利刃,PvE,zhCN,CN,Daggerspine",
[1659]="深渊之巢,PvE,zhCN,CN,Deephome",
[1660]="埃基尔松,PvE,zhCN,CN,Akil'zon",
[1662]="火烟之谷,PvE,zhCN,CN,Dustbelcher",
[1663]="伊兰尼库斯,PvE,zhCN,CN,Eranikus",
[1664]="火喉,PvE,zhCN,CN,Firegut",
[1666]="冬寒,PvE,zhCN,CN,Winterchill",
[1667]="迦玛兰,PvE,zhCN,CN,Jammal'an",
[1668]="金度,PvE,zhCN,CN,Jin'do",
[1670]="巫妖之王,PvE,zhCN,CN,Lich King",
[1672]="米奈希尔,PvE,zhCN,CN,Menethil",
[1676]="幽暗沼泽,PvE,zhCN,CN,The Underbog",
[1681]="烈焰荆棘,PvE,zhCN,CN,Smolderthorn",
[1682]="夺灵者,PvE,zhCN,CN,Soulflayer",
[1685]="石锤,PvE,zhCN,CN,Stonemaul",
[1687]="塞拉摩,PvE,zhCN,CN,Theramore",
[1692]="厄祖玛特,PvE,zhCN,CN,Ozumat",
[1693]="冬泉谷,PvE,zhCN,CN,Winterspring",
[1694]="伊森德雷,PvE,zhCN,CN,Ysondre",
[1695]="扎拉赞恩,PvE,zhCN,CN,Zalazane",
[1696]="亚雷戈斯,PvE,zhCN,CN,Arygos",
[1793]="深渊之喉,PvE,zhCN,CN,Abyssal Maw",
[1794]="凤凰之神,PvE,zhCN,CN,Al'ar",
[1795]="阿古斯,PvE,zhCN,CN,Argus",
[1796]="奥金顿,PvE,zhCN,CN,Auchindoun",
[1797]="刀塔,PvE,zhCN,CN,Bladespire",
[1798]="鲜血熔炉,PvE,zhCN,CN,Blood Furnace",
[1801]="黑暗之门,PvE,zhCN,CN,Dark Portal",
[1802]="死亡熔炉,PvE,zhCN,CN,Deathforge",
[1803]="无底海渊,PvE,zhCN,CN,Abyssal Depths",
[1807]="格鲁尔,PvE,zhCN,CN,Gruul",
[1808]="哈兰,PvE,zhCN,CN,Halaa",
[1809]="军团要塞,PvE,zhCN,CN,Legion Hold",
[1810]="麦姆,PvE,zhCN,CN,Maim",
[1812]="艾露恩,PvE,zhCN,CN,Elnue",
[1813]="穆戈尔,PvE,zhCN,CN,Mogor",
[1815]="摩摩尔,PvE,zhCN,CN,Murmur",
[1817]="试炼之环,PvE,zhCN,CN,Ring of Trials",
[1818]="罗曼斯,PvE,zhCN,CN,Rommath",
[1819]="希雷诺斯,PvE,zhCN,CN,Searinox",
[1820]="塞泰克,PvE,zhCN,CN,Sethekk",
[1821]="暗影迷宫,PvE,zhCN,CN,Shadow Labyrinth",
[1823]="托尔巴拉德,PvE,zhCN,CN,Tol Barad",
[1824]="太阳之井,PvE,zhCN,CN,Sunwell",
[1827]="末日祷告祭坛,PvE,zhCN,CN,Doom's Vigil",
[1828]="范克里夫,PvE,zhCN,CN,VanCleef",
[1829]="瓦丝琪,PvE,zhCN,CN,Vashj",
[1830]="祖阿曼,PvE,zhCN,CN,Zul'Aman",
[1831]="祖达克,PvE,zhCN,CN,Zul'Drak",
[1832]="翡翠梦境,PvE,zhCN,CN,Emerald Dream",
[1922]="Азурегос,PvE,ruRU,EU,Azuregos",
[1923]="Ясеневый лес,PvE,ruRU,EU,Ashenvale",
[1924]="Пиратская Бухта,PvE,ruRU,EU,Booty Bay",
[1925]="Вечная Песня,PvE,ruRU,EU,Eversong",
[1926]="Термоштепсель,PvE,ruRU,EU,Thermaplugg",
[1927]="Гром,PvE,ruRU,EU,Grom",
[1928]="Голдринн,PvE,ruRU,EU,Goldrinn",
[1929]="Черный Шрам,PvE,ruRU,EU,Blackscar",
[1931]="阿比迪斯,PvE,zhCN,CN,Abbendis",
[1932]="阿曼尼,PvE,zhCN,CN,Amani",
[1933]="安苏,PvE,zhCN,CN,Anzu",
[1934]="生态船,PvE,zhCN,CN,The Botanica",
[1935]="阿斯塔洛,PvE,zhCN,CN,Astalor",
[1936]="白骨荒野,PvE,zhCN,CN,Bone Wastes",
[1937]="布鲁塔卢斯,PvE,zhCN,CN,Brutallus",
[1938]="达尔坎,PvE,zhCN,CN,Dar'Khan",
[1939]="末日行者,PvE,zhCN,CN,Doomwalker",
[1940]="达基萨斯,PvE,zhCN,CN,Drakkisath",
[1941]="熵魔,PvE,zhCN,CN,Entropius",
[1942]="能源舰,PvE,zhCN,CN,The Mechanar",
[1943]="菲米丝,PvE,zhCN,CN,Felmyst",
[1944]="加尔,PvE,zhCN,CN,Garr",
[1945]="迦顿,PvE,zhCN,CN,Geddon",
[1946]="血吼,PvE,zhCN,CN,Gorehowl",
[1947]="戈提克,PvE,zhCN,CN,Gothik",
[1948]="盖斯,PvE,zhCN,CN,Gyph",
[1949]="壁炉谷,PvE,zhCN,CN,Hearthglen",
[1950]="贫瘠之地,PvE,zhCN,CN,Barrens",
[1955]="霍格,PvE,zhCN,CN,Hogger",
[1965]="奎尔丹纳斯,PvE,zhCN,CN,Que'Danis",
[1969]="萨洛拉丝,PvE,zhCN,CN,Sacrolash",
[1970]="沙怒,PvE,zhCN,CN,Sandfury",
[1971]="嚎风峡湾,PvE,zhCN,CN,Howling Fjord",
[1972]="斯克提斯,PvE,zhCN,CN,Skettis",
[2075]="雲蛟衛,PvE,zhTW,TW,Order of the Cloud Serpent",
[2079]="와일드해머,PvE,koKR,KR,Wildhammer",
[2106]="렉사르,PvE,koKR,KR,Rexxar",
[2107]="하이잘,PvE,koKR,KR,Hyjal",
[2108]="데스윙,PvE,koKR,KR,Deathwing",
[2110]="세나리우스,PvE,koKR,KR,Cenarius",
[2111]="스톰레이지,PvE,koKR,KR,Stormrage",
[2116]="줄진,PvE,koKR,KR,Zul'jin",
[2118]="迦拉克隆,PvE,zhCN,CN,Galakrond",
[2119]="奥杜尔,PvE,zhCN,CN,Ulduar",
[2120]="奥尔加隆,PvE,zhCN,CN,Algalon",
[2121]="安格博达,PvE,zhCN,CN,Angerboda",
[2122]="安加萨,PvE,zhCN,CN,Angrathar",
[2123]="织亡者,PvE,zhCN,CN,Deathweaver",
[2124]="亡语者,PvE,zhCN,CN,Deathwhisper",
[2125]="达克萨隆,PvE,zhCN,CN,Drak'Tharon",
[2126]="黑锋哨站,PvE,zhCN,CN,Ebon Watch",
[2127]="古达克,PvE,zhCN,CN,Gundrak",
[2128]="兰娜瑟尔,PvE,zhCN,CN,Lana'thel",
[2129]="洛肯,PvE,zhCN,CN,Loken",
[2130]="玛洛加尔,PvE,zhCN,CN,Marrowgar",
[2133]="影之哀伤,PvE,zhCN,CN,Shadowmourne",
[2134]="风暴峭壁,PvE,zhCN,CN,Storm Peaks",
[2135]="远古海滩,PvE,zhCN,CN,Strand of the Ancients",
[2136]="瓦拉纳,PvE,zhCN,CN,Valanar",
[2137]="冬拥湖,PvE,zhCN,CN,Wintergrasp",
[3207]="Goldrinn,PvE,ptBR,US,BRT",
[3208]="Nemesis,PvE,ptBR,US,US,BRT",
[3209]="Azralon,PvE,ptBR,US,US,BRT",
[3210]="Tol Barad,PvE,ptBR,US,US,BRT",
[3234]="Gallywix,PvE,ptBR,US,US,BRT",
[3721]="Caelestrasz,PvE,enUS,US,AEST",
[3722]="Aman'Thul,PvE,enUS,US,AEST",
[3723]="Barthilas,PvE,enUS,US,AEST",
[3724]="Thaurissan,PvE,enUS,US,AEST",
[3725]="Frostmourne,PvE,enUS,US,AEST",
[3726]="Khaz'goroth,PvE,enUS,US,AEST",
[3733]="Dreadmaul,PvE,enUS,US,AEST",
[3734]="Nagrand,PvE,enUS,US,AEST",
[3735]="Dath'Remar,PvE,enUS,US,AEST",
[3736]="Jubei'Thos,PvE,enUS,US,AEST",
[3737]="Gundrak,PvE,enUS,US,AEST",
[3738]="Saurfang,PvE,enUS,US,AEST",
[3751]="丽丽（四川),PvE,zhCN,CN,Li Li",
[3752]="晴日峰（江苏),PvE,zhCN,CN,Peak of Serenity",
[3755]="辛达苟萨,PvE,zhCN,CN,Sindragosa",
[3757]="时光之穴,PvE,zhCN,CN,Caverns of Time",
[3941]="苏拉玛,PvE,zhCN,CN,Suramar",
[3944]="瓦里安,PvE,zhCN,CN,Varian",
	-- {{ Classic Realm Data, Manually Compiled. I take no responsibility for inaccurate/Outdated information, Please report any inconsistencies to me :) Translated Chinese Names may also differ from what Blizzard calls the server.
[4372]="Atiesh,PvE,enUS,US,PST",
[4373]="Mzrael,PvE,enUS,US,PST",
[4374]="Old Blanchy,PvE,enUS,US,PST",
[4376]="Azuresong,PvE,enUS,US,PST",
[4384]="Mankrik,PvE,enUS,US,EST",
[4385]="Pagle,PvE,enUS,US,EST",
[4386]="Deviate Delight,PvP RP,enUS,US,EST",
[4387]="Thunderfury,PvE,enUS,US,EST",
[4388]="Westfall,PvE,enUS,US,EST",
[4395]="Whitemane,PvP,enUS,US,PST",
[4396]="Fairbanks,PvP,enUS,US,PST",
[4397]="Blaumeux,PvP,enUS,US,PST",
[4398]="Bigglesworth,PvP,enUS,US,PST",
[4399]="Kurinaxx,PvP,enUS,US,PST",
[4406]="Herod,PvP,enUS,US,EST",
[4407]="Thalnos,PvP,enUS,US,EST",
[4408]="Faerlina,PvP,enUS,US,EST",
[4409]="Stalegg,PvP,enUS,US,EST",
[4410]="Skeram,PvP,enUS,US,EST",
[4417]="소금 평원,PvE,koKR,KR,Shimmering Flats",
[4419]="로크홀라,PvP,koKR,KR,Lokholar",
[4420]="얼음피,PvP,koKR,KR,Iceblood",
[4421]="라그나로스,PvP,koKR,KR,Ragnaros",
[4440]="Everlook,PvE,deDE,EU",
[4441]="Auberdine,PvE,frFR,EU",
[4442]="Lakeshire,PvE,deDE,EU",
[4452]="Хроми,PvE,ruRU,EU,Chromie",
[4453]="Pyrewood Village,PvE,enUS,EU",
[4455]="Razorfen,PvE,deDE,EU",
[4456]="Nethergarde Keep,PvE,enUS,EU",
[4458]="Mirage Raceway,PvE,enUS,EU",
[4463]="Heartstriker,PvP,deDE,EU",
[4464]="Sulfuron,PvP,frFR,EU",
[4465]="Golemagg,PvP,enUS,EU",
[4466]="Patchwerk,PvP,deDE,EU",
[4467]="Firemaw,PvP,enUS,EU",
[4474]="Пламегор,PvP,ruRU,EU,Flamegor",
[4475]="Shazzrah,PvP,enUS,EU",
[4476]="Gehennas,PvP,enUS,EU",
[4477]="Venoxis,PvP,deDE,EU",
[4478]="Razorgore,PvP,enUS,EU",
[4485]="瑪拉頓,PvE,zhTW,TW,Maraudon",
[4487]="伊弗斯,PvP,zhTW,TW,Ivus",
[4497]="碧玉矿洞,PvE,zhCN,CN,Azurelode Mine",
[4498]="寒脊山小径,PvE,zhCN,CN,Coldridge Pass",
[4499]="埃提耶什,PvP,zhCN,CN,Atiesh",
[4500]="龙之召唤,PvP,zhCN,CN,Dragon's Call",
[4501]="加丁,PvP,zhCN,CN,Gandling",
[4509]="哈霍兰,PvP,zhCN,CN,Huhuran",
[4510]="奥罗,PvP,zhCN,CN,Ouro",
[4511]="沙尔图拉,PvP,zhCN,CN,Sartura",
[4512]="莫格莱尼,PvP,zhCN,CN,Mograine",
[4513]="希尔盖,PvP,zhCN,CN,Heigan",
[4520]="匕首岭,PvE,zhCN,CN,The Dagger Hills",
[4521]="厄运之槌,PvE,zhCN,CN,Dire Maul",
[4522]="雷霆之击,PvP,zhCN,CN,Thunderstrike",
[4523]="法尔班克斯,PvP,zhCN,CN,Fairbanks",
[4524]="赫洛德,PvP,zhCN,CN,Herod",
[4531]="吞咽者,PvP,zhCN,CN,Buru",
[4532]="范克瑞斯,PvP,zhCN,CN,Fankriss",
[4533]="维希度斯,PvP,zhCN,CN,Viscidus",
[4534]="帕奇维克,PvP,zhCN,CN,Patchwerk",
[4535]="比格沃斯,PvP,zhCN,CN,Bigglesworth",
[4647]="Grobbulus,PvP RP,enUS,US,PST",
[4648]="Bloodsail Buccaneers,RP,enUS,US,EST",
[4667]="Remulos,PvE,enUS,US,AEST",
[4669]="Arugal,PvP,enUS,US,AEST",
[4670]="Yojamba,PvP,enUS,US,AEST",
[4674]="木喉要塞,RP,zhCN,CN,Timbermaw Hold",
[4675]="辛迪加,PvP RP,zhCN,CN,Syndicate",
[4676]="Zandalar Tribe,PvP RP,enUS,EU",
[4678]="Hydraxian Waterlords,RP,enUS,EU",
[4695]="Rattlegore,PvP,enUS,US,PST",
[4696]="Smolderweb,PvP,enUS,US,PST",
[4698]="Incendius,PvP,enUS,US,EST",
[4699]="Kromcrush,PvP,enUS,US,EST",
[4700]="Kirtonos,PvP,enUS,US,EST",
[4701]="Mograine,PvP,enUS,EU",
[4702]="Gandling,PvP,enUS,EU",
[4703]="Amnennar,PvP,frFR,EU",
[4704]="Змейталак,PvP,ruRU,EU,Wyrmthalak",
[4705]="Stonespire,PvP,enUS,EU",
[4706]="Flamelash,PvP,enUS,EU",
[4707]="霜语,PvP,zhCN,CN,Frostwhisper",
[4708]="水晶之牙,PvP,zhCN,CN,Crystal Fang",
[4709]="维克洛尔,PvP,zhCN,CN,Vek'lor",
[4710]="维克尼拉斯,PvP,zhCN,CN,Vek'nilash",
[4711]="巴罗夫,PvP,zhCN,CN,Barov",
[4712]="比斯巨兽,PvP,zhCN,CN,The Beast",
[4714]="Thunderfury,PvP,enUS,US,PST",
[4715]="Anathema,PvP,enUS,US,PST",
[4716]="Arcanite Reaper,PvP,enUS,US,PST",
[4726]="Sulfuras,PvP,enUS,US,EST",
[4727]="Windseeker,PvE,enUS,US,EST",
[4728]="Benediction,PvP,enUS,US,EST",
[4729]="Netherwind,PvP,enUS,US,EST",
[4731]="Earthfury,PvP,enUS,US,EST",
[4732]="Heartseeker,PvP,enUS,US,EST",
[4737]="Sul'thraze,PvP,ptBR,US,BRT",
[4739]="Felstriker,PvP,enUS,US,AEST",
[4741]="Noggenfogger,PvP,enUS,EU",
[4742]="Ashbringer,PvP,enUS,EU",
[4743]="Skullflame,PvP,enUS,EU",
[4744]="Finkle,PvP,frFR,EU",
[4745]="Transcendence,PvP,deDE,EU",
[4746]="Bloodfang,PvP,enUS,EU",
[4749]="Earthshaker,PvP,enUS,EU",
[4751]="Dragonfang,PvP,enUS,EU",
[4754]="Рок-Делар,PvP,ruRU,EU,Rhok'delar",
[4755]="Dreadmist,PvP,enUS,EU",
[4756]="Dragon's Call,PvP,deDE,EU",
[4757]="Ten Storms,PvP,enUS,EU",
[4758]="Judgement,PvP,enUS,EU",
[4759]="Celebras,RP,deDE,EU",
[4763]="Heartstriker,PvP,deDE,EU",
[4766]="Вестник Рока,PvP,ruRU,EU,Harbinger of Doom",
[4767]="诺格弗格,PvP,zhCN,CN,Noggenfogger",
[4768]="毁灭之刃,PvP,zhCN,CN,Perdition's Blade",
[4769]="黑曜石之锋,PvP,zhCN,CN,Obsidian Edge",
[4770]="萨弗拉斯,PvP,zhCN,CN,Sulfuras",
[4771]="伦鲁迪洛尔,PvP,zhCN,CN,Rhok'delar",
[4772]="灰烬使者,PvP,zhCN,CN,Ashbringer",
[4773]="怀特迈恩,PvP,zhCN,CN,Whitemane",
[4774]="奥金斧,PvP,zhCN,CN,Arcanite Reaper",
[4775]="骨火,PvP,zhCN,CN,Skullflame",
[4776]="末日之刃,PvP,zhCN,CN,Doom's Edge",
[4777]="震地者,PvP,zhCN,CN,Earthshaker",
[4778]="祈福,PvP,zhCN,CN,Benediction",
[4779]="辛洛斯,PvP,zhCN,CN,Zin'rokh",
[4780]="觅心者,PvP,zhCN,CN,Heartseeker",
[4781]="狮心,PvP,zhCN,CN,Lionheart",
[4782]="审判,PvP,zhCN,CN,Judgement",
[4783]="无尽风暴,PvP,zhCN,CN,Ten Storms",
[4784]="巨龙追猎者,PvP,zhCN,CN,Dragonstalker",
[4785]="灵风,PvP,zhCN,CN,Netherwind",
[4786]="卓越,PvP,zhCN,CN,Transcendence",
[4787]="狂野之刃,PvP,zhCN,CN,The Untamed Blade",
[4788]="巨人追猎者,PvP,zhCN,CN,Giantstalker",
[4789]="秩序之源,PvP,zhCN,CN,Lawbringer",
[4790]="奎尔塞拉,PvP,zhCN,CN,Quel'Serrar",
[4791]="碧空之歌,PvP,zhCN,CN,Azuresong",
[4792]="阿什坎迪,PvP,zhCN,CN,Ashkandi",
[4801]="Loatheb,PvP,esMX,US,EST",
[4813]="Mandokir,PvP,esES,EU",
[4818]="艾隆纳亚,PvP,zhCN,CN,Ironara",
[4819]="席瓦莱恩,PvP,zhCN,CN,Silverlaine",
[4820]="火锤,PvP,zhCN,CN,Firehammer",
[4821]="沙顶,PvP,zhCN,CN,Sandscalp",
[4822]="德姆塞卡尔,PvP,zhCN,CN,Dreamscythe",
[4823]="吉兹洛克,PvP,zhCN,CN,Gizlock",
[4824]="怒炉,PvP,zhCN,CN,Angerforge",
[4825]="法拉克斯,PvE,zhCN,CN,Phalanx",
[4826]="乌洛克,PvP,zhCN,CN,Urok",
[4827]="无畏,PvP,zhCN,CN,Dreadnaught",
[4828]="拉姆斯登,PvP,zhCN,CN,Ramstein",
[4829]="安娜丝塔丽,PvP,zhCN,CN,Anastari",
[4830]="诺克赛恩,PvP,zhCN,CN,Noxxion",
[4831]="塞雷布拉斯,PvP,zhCN,CN,Cerebras",
[4832]="雷德,PvP,zhCN,CN,Rend",
[4833]="曼多基尔,PvP,zhCN,CN,Mandokir",
[4834]="娅尔罗,PvP,zhCN,CN,Arlokk",
[4835]="塞卡尔,PvP,zhCN,CN,Thekal",
[4836]="迈克斯纳,PvP,zhCN,CN,Maexxna",
[4837]="范沃森,PvP,zhCN,CN,VanHowzen",
[4839]="힐스브래드,PvP,koKR,KR,Hillsbrad",
[4847]="光芒,PvP,zhCN,CN,Flarecore",
[4848]="萨弗隆,PvP,zhCN,CN,Sulfuron",
[4849]="维克托,PvP,zhCN,CN,Viktori",
[4913]="寒冰之王,PvP,zhCN,CN,Coldbringer",
[4915]="布劳缪克丝,PvP,zhCN,CN,Blaumeux",
[4919]="阿鲁高,PvP,zhCN,CN,Arugal",
[4920]="龙牙,PvP,zhCN,CN,Dragonfang",
[4923]="弗莱拉斯,PvP,zhCN,CN,Flamelash",
[4924]="法琳娜,PvP,zhCN,CN,Faerlina",
[4925]="湖畔镇,PvE,zhCN,CN,Lakeshire",
[4926]="克罗米,PvP,zhCN,CN,Chromie",
[4928]="伊森迪奥斯,PvP,zhCN,CN,Incendius",
[4970]="阿拉希盆地,PvP,zhCN,CN,Arathi Basin",
[4972]="巨龙沼泽,PvP,zhCN,CN,Wyrmbog",
[4973]="黑石山,PvP,zhCN,CN,Blackrock Mountain",
[4974]="圣光之愿,PvP,zhCN,CN,Light's Hope",
[4975]="神谕林地,PvP,zhCN,CN,The Enchanted Glade",
[4976]="流沙岗哨,PvP,zhCN,CN,Sandsorrow Watch",
[4977]="甲虫之墙,PvP,zhCN,CN,The Scarab Wall",
[4978]="祖尔格拉布,PvP,zhCN,CN,Zul'Gurub",
}

connectionData = {
"4,US,4,1355",
"5,US,5",
"9,US,9",
"11,US,11",
"12,US,12,1154,1370",
"47,US,47,1354",
"52,US,52,62,65,1565",
"53,US,53,82,92,93,159,1572",
"54,US,54,1581",
"55,US,55,1271",
"57,US,57",
"58,US,58",
"60,US,60",
"61,US,61",
"63,US,63,1270",
"64,US,64,1258",
"67,US,67,97,101,1067,1555",
"69,US,69",
"71,US,71,80,1075,1288,1293,1294,1344",
"73,US,73",
"75,US,75,1570",
"76,US,76",
"77,US,77,70,79,103,1128,1131,1558",
"78,US,78,1264,1268,1282",
"84,US,84,116,123,1137,1145,1349,1363",
"86,US,86,85,90,131,1140,1368,1371,1374",
"96,US,96,74,156,1068,1259,1267,1276,1280,1567",
"99,US,99,1297",
"100,US,100",
"104,US,104,14,51,91,95,1285",
"106,US,106,1576",
"113,US,113,87,1139,1351",
"114,US,114,1345",
"115,US,115,1342",
"117,US,117,1364",
"118,US,118,126",
"120,US,120",
"121,US,121,10,1143,1182,1359",
"125,US,125,88,1347,1356",
"127,US,127,7,110,124,1148,1348,1358,1362",
"151,US,151,3",
"154,US,154,81,94,1266,1295,1296",
"155,US,155,15,1277,1557",
"157,US,157,72,1278,1286,1556",
"158,US,158,1292",
"160,US,160,1173,1549,1582",
"162,US,162",
"163,US,163,164,1130,1289,1352",
"205,KR,205",
"210,KR,210,201,2111",
"214,KR,214,258,2079,2106,2108",
"509,EU,509,544,546",
"510,EU,510,1620",
"512,EU,512,543,642,643",
"531,EU,531,605,610,615,1319",
"570,EU,570,565,579,616",
"578,EU,578,535,588,609,1318,1328,1613",
"580,EU,580,1106,1409",
"581,EU,581",
"604,EU,604,568,1330",
"612,EU,612,567,582,586,591,611,1323",
"633,EU,633,556,630,1087,1314,1389,1392,1415",
"705,CN,705,835",
"706,CN,706,922,925,1501",
"707,CN,707",
"708,CN,708,753,822,867,1213,1815,1941",
"709,CN,709,879,1201",
"714,CN,714",
"721,CN,721,755,767,807,865,926,1203,1215",
"723,CN,723,717,740,1231,1499",
"727,CN,727,1235",
"729,CN,729",
"730,CN,730,739,1482",
"736,CN,736,843,847,855,1200,1666,1797,1803",
"738,CN,738,765",
"741,CN,741",
"743,CN,743,710,790,1817",
"744,CN,744,827,2130",
"745,CN,745,782,1202,1237",
"746,CN,746,754,780,792",
"797,CN,797",
"799,CN,799,1828",
"802,CN,802,732,815,943",
"803,CN,803,720,932,1935",
"804,CN,804,716,805,861,1818,1820",
"810,CN,810,812,825,1827",
"814,CN,814,771,1513",
"816,CN,816",
"817,CN,817,860,1664,1795",
"826,CN,826,776,787",
"829,CN,829,866,1222",
"830,CN,830,1502",
"833,CN,833,1212",
"838,CN,838",
"839,CN,839,1496",
"840,CN,840,726,742,1490",
"842,CN,842",
"844,CN,844,1831",
"845,CN,845,748",
"846,CN,846,1236",
"848,CN,848",
"850,CN,850",
"851,CN,851,1823",
"852,CN,852",
"857,CN,857,1487",
"858,CN,858,1693",
"859,CN,859,757",
"863,CN,863,751,1520,1657",
"869,CN,869,711,1662",
"870,CN,870,791",
"872,CN,872,762,770",
"876,CN,876,1667",
"877,CN,877,718,887,941",
"878,CN,878,1234,1807,1813",
"882,CN,882,849",
"883,CN,883,924",
"886,CN,886",
"889,CN,889,731,841,856",
"891,CN,891,703",
"915,CN,915",
"916,CN,916,756",
"917,CN,917,1969",
"918,CN,918,874",
"920,CN,920,750,1676,1798",
"921,CN,921",
"929,CN,929,1216",
"938,CN,938,1486,1512",
"940,CN,940",
"951,CN,951,1830",
"953,CN,953,818",
"954,CN,954,764,800,1238,1484,1503",
"956,CN,956,808,1832",
"959,CN,959,725,837,949,1524,1681,1819",
"962,CN,962",
"963,TW,963,964,979,999,1001,1033,1054,1056,1057",
"966,TW,966,965,977,982,1006,1037,1038,1043,2075",
"980,TW,980,978,985,1023,1046,1048,1049",
"1070,US,1070,1563",
"1071,US,1071,1260,1290",
"1072,US,1072,1283",
"1080,EU,1080,504",
"1082,EU,1082,1391,1394",
"1084,EU,1084,1306",
"1085,EU,1085,1117,1595",
"1091,EU,1091,513,518,522,525,646",
"1092,EU,1092,523",
"1096,EU,1096,561,635,636,1308,1317,1606",
"1097,EU,1097,1324",
"1098,EU,1098,572",
"1099,EU,1099,563",
"1104,EU,1104,587,589,594,1322,1611",
"1105,EU,1105,573,584,601,608,1321",
"1121,EU,1121,576,613,614,1118,1119,1326,1619",
"1122,EU,1122,641,1123,1332",
"1127,EU,1127,537,644,647,1086,1337,1626",
"1129,US,1129,56,98,102,105,1262,1291,1559,1564",
"1136,US,1136,83,109,129,1142,1373",
"1138,US,1138,89,107,130,1141,1165,1346,1377",
"1147,US,1147,1353",
"1151,US,1151,1350",
"1168,US,2,8,122,128,1146,1360,1361",
"1171,US,1369",
"1175,US,1175,68,153,1069,1132,1287,1578,1579",
"1184,US,1184,1367,1375",
"1185,US,1185,108,111,112,119,1357,1372",
"1190,US,1190,13",
"1301,EU,1301",
"1302,EU,539",
"1303,EU,1303,1300,1413",
"1305,EU,1305",
"1307,EU,1307",
"1309,EU,1309",
"1313,EU,1313,552",
"1315,EU,540,645",
"1316,EU,1316",
"1325,EU,500,619",
"1329,EU,554",
"1331,EU,1331,517",
"1335,EU,1335",
"1378,EU,1378,1381",
"1379,EU,1379,1380,1382,1383",
"1384,EU,1384,1387,1395",
"1385,EU,1385,1386",
"1388,EU,1388,1089",
"1390,EU,542",
"1393,EU,1393,618",
"1396,EU,503,623",
"1400,EU,1400,602,1404",
"1401,EU,1401,574,575,1407,1608",
"1402,EU,505,553",
"1403,EU,506",
"1405,EU,1405,516,592,617,1327,1333",
"1406,EU,1406,569",
"1408,EU,1408,600",
"1416,EU,1416,1081,1298,1310,1312",
"1417,EU,1417,550",
"1425,US,1425",
"1426,US,1284",
"1427,US,1427",
"1428,US,1428",
"1483,CN,1483,786",
"1485,CN,1485,1812",
"1488,CN,1488,749,1663",
"1489,CN,1489,1227",
"1491,CN,1491",
"1492,CN,1492,885,930",
"1493,CN,1493,772,927,1796,1808",
"1494,CN,1494,1209",
"1495,CN,1495,1937,1965",
"1498,CN,1498,1516",
"1500,CN,1500,734,806,1204,1497,1829",
"1504,CN,1504,1206,1224,1802",
"1505,CN,1505,768,1239",
"1506,CN,1506,793,1682",
"1509,CN,1509,778,946,1199",
"1510,CN,1510,1230",
"1514,CN,1514,1824",
"1515,CN,1515,1794",
"1517,CN,1517,700,788,1214,1223,1507,1659,1793,2123",
"1519,CN,1519,1208,1229,1660,1692",
"1587,EU,1587,501,547,1311,1589",
"1596,EU,1596,507,527,627,637,1588",
"1597,EU,1597,628,529,1304",
"1598,EU,1598,511,526,607,621,1083,1088,1090,1093,1299",
"1602,EU,1602",
"1604,EU,1604",
"1605,EU,1605",
"1614,EU,1614,1609,1616",
"1615,EU,1615",
"1618,EU,1618",
"1621,EU,1621,533,538,545,1336",
"1623,EU,1623",
"1624,EU,1624,541,1334,1622",
"1668,CN,1668,1207",
"1670,CN,1670,719,1945",
"1672,CN,1672,1240",
"1687,CN,1687,1810,1821",
"1694,CN,1694,760,1211,1226",
"1695,CN,1695,704",
"1696,CN,1696,888",
"1922,EU,1922",
"1923,EU,1923",
"1925,EU,1925",
"1928,EU,1928,1603,1610",
"1929,EU,1929,1617,1625,1924,1926,1927",
"1931,CN,1931,933",
"1932,CN,1932,761,1228",
"1933,CN,1933",
"1934,CN,1934,1809",
"1936,CN,1936,828,1658,1942",
"1938,CN,1938,890,1233,1685",
"1939,CN,1939",
"1943,CN,1943,781,944",
"1944,CN,1944,715",
"1946,CN,1946,1801",
"1948,CN,1948,766,1198",
"1949,CN,1949",
"1950,CN,1950",
"1955,CN,1955,794",
"1970,CN,1970,773,774,931,960,1205,1232,1508,1947",
"1971,CN,1971,784",
"1972,CN,1972",
"2073,EU,560",
"2074,EU,508,551",
"2116,KR,2116,207,211,212,215,264,293,2107,2110",
"2118,CN,2118",
"2119,CN,2119,737,1511",
"2121,CN,2121,758",
"2122,CN,2122,1241",
"2124,CN,2124,864,2120",
"2125,CN,2125,1210",
"2126,CN,2126,1221",
"2127,CN,2127,775",
"2128,CN,2128,769",
"2129,CN,2129,1225",
"2133,CN,2133",
"2134,CN,2134",
"2135,CN,2135",
"2136,CN,2136",
"2137,CN,2137,712,1940",
"3207,US,3207",
"3208,US,3208,3210",
"3209,US,3209",
"3234,US,3234",
"3391,EU,549",
"3656,EU,528,558,559,629,638",
"3657,EU,515,521,606,624,631,632",
"3661,US,6",
"3666,EU,502,548",
"3674,EU,625",
"3675,US,1365",
"3676,US,1566",
"3678,US,1263",
"3679,EU,577",
"3681,EU,622",
"3682,EU,626",
"3683,US,66",
"3684,US,59",
"3685,US,1265",
"3686,EU,564",
"3690,EU,640",
"3691,EU,566,590,1320,1612",
"3692,EU,583",
"3693,US,16",
"3694,US,1",
"3696,EU,562,571,593,1607",
"3702,EU,536",
"3703,EU,585",
"3713,EU,519,524,557,639",
"3721,US,3721,3734,3738",
"3723,US,3723",
"3725,US,3725,3724,3733,3736,3737",
"3726,US,3726,3722,3735",
"3751,CN,3751",
"3752,CN,3752",
"3755,CN,3755,862",
"3757,CN,3757",
"3941,CN,3941",
"3944,CN,3944",
}

------------------------------------------------------------------------

if standalone then
	LRI_RealmData = realmData
	LRI_ConnectionData = connectionData
end
