--[[-------------------------------------------------------------------
--  Clique - Copyright 2006-2024 - James N. Whitehead II
-------------------------------------------------------------------]] ---

---@class addon
local addon = select(2, ...)
local L = addon.L

---@class BindingConfig
local config = addon:GetBindingConfig()

local page = {}

-- Globals used in this file
local PlaySound = PlaySound
local SOUNDKIT = SOUNDKIT

function config:GetEditPage()
    return page
end

function page:Show()
    page.frame:Show()
end

function page:IsShown()
    return page.frame:IsShown()
end

function page:Hide()
    page.frame:Hide()
end

function page:Initialize()
    if page.initialized then
        return
    end

    page.initialized = true

    page.frame = CreateFrame("Frame", "CliqueConfigUIBindingFrameEditPage", config.ui)
    local frame = page.frame

    frame:SetAllPoints()
    frame:Hide()

    frame.SaveButton = CreateFrame("Button", nil, frame, "UIPanelButtonTemplate")
    frame.SaveButton:SetText(L["保存绑定"])
    frame.SaveButton:SetHeight(23)
    frame.SaveButton:SetWidth(120)
    frame.SaveButton:ClearAllPoints()
    frame.SaveButton:SetPoint("BOTTOMRIGHT", frame, "BOTTOMRIGHT", -6, 5)
    frame.SaveButton:SetScript("OnClick", function(self, button)
        page:SaveButton_OnClick(self, button)
    end)

    frame.CancelButton = CreateFrame("Button", nil, frame, "UIPanelButtonTemplate")
    frame.CancelButton:SetText(L["取消"])
    frame.CancelButton:SetHeight(23)
    frame.CancelButton:SetWidth(120)
    frame.CancelButton:ClearAllPoints()
    frame.CancelButton:SetPoint("RIGHT", frame.SaveButton, "LEFT", 0, 0)
    frame.CancelButton:SetScript("OnClick", function(self, button)
        page:CancelButton_OnClick(self, button)
    end)

    frame.RemoveRankButton = CreateFrame("Button", nil, frame, "UIPanelButtonTemplate")
    frame.RemoveRankButton:SetText(L["删除排名/学校"])
    frame.RemoveRankButton:SetHeight(23)
    frame.RemoveRankButton:SetWidth(200)
    frame.RemoveRankButton:ClearAllPoints()
    frame.RemoveRankButton:SetPoint("BOTTOMLEFT", frame, "BOTTOMLEFT", 6, 5)
    frame.RemoveRankButton:Hide()
    frame.RemoveRankButton:SetScript("OnClick", function(self, button)
        page:RemoveRankButton_OnClick(self, button)
    end)

    frame.bindSummary = CreateFrame("Button", nil, frame, "CliqueBindingSummaryTemplate")
    frame.bindSummary:SetWidth(200)
    frame.bindSummary:SetHeight(100)
    frame.bindSummary.Icon:SetTexture(132212)
    frame.bindSummary.Name:SetText("Name")
    frame.bindSummary.Text:SetText("Some text")
    frame.bindSummary.BindingText:SetText("Binding text")

    frame.bindSummary:ClearAllPoints()
    frame.bindSummary:SetPoint("TOPLEFT", frame, "TOPLEFT", 15, -65)
    frame.bindSummary:SetPoint("TOPRIGHT", frame, "TOPRIGHT", -15, -65)

    frame.changeBinding = CreateFrame("Button", nil, frame, "UIMenuButtonStretchTemplate")
    frame.changeBinding:SetText(L["更改绑定"])
    frame.changeBinding:ClearAllPoints()
    frame.changeBinding:SetPoint("TOPRIGHT", frame.bindSummary, "BOTTOMRIGHT", 0, -5)
    frame.changeBinding:SetWidth(125)

    -- Glow box for when changing a binding
    frame.changeBindingArrow = CreateFrame("Frame", nil, frame, "GlowBoxArrowTemplate")
    frame.changeBindingArrow:ClearAllPoints()
    frame.changeBindingArrow:SetPoint("TOP", frame.changeBinding, "BOTTOM", -10, -2)

    frame.changeBindingHelpBox = CreateFrame("Frame", nil, frame.changeBindingArrow, "GlowBoxTemplate")
    frame.changeBindingHelpBox:SetWidth(250)
    frame.changeBindingHelpBox:SetHeight(120)
    frame.changeBindingHelpBox:ClearAllPoints()
    frame.changeBindingHelpBox:SetPoint("TOPLEFT", frame.changeBindingArrow, "BOTTOMLEFT", 0, -5)
    frame.changeBindingHelpBox.Text = frame.changeBindingHelpBox:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
    frame.changeBindingHelpBox.Text:SetJustifyH("LEFT")
    frame.changeBindingHelpBox.Text:SetJustifyV("MIDDLE")
    frame.changeBindingHelpBox.Text:SetPoint("TOPLEFT", 10, -5)
    frame.changeBindingHelpBox.Text:SetPoint("BOTTOMRIGHT", -10, 5)

    local changeBindingHelptext = L["您正处于绑定捕获模式！您可以使用鼠标单击或按键盘上的某个键来设置绑定。您可以通过按住键盘上的 alt、ctrl 和 shift 键的组合来修改绑定。"]
    frame.changeBindingHelpBox.Text:SetText(changeBindingHelptext)
    frame.changeBindingHelpBox:SetFrameStrata("HIGH")
    frame.changeBindingArrow:Hide()

    -- Glow box for when creating a new binding
    frame.addBindingHelpBox = CreateFrame("Frame", nil, frame.bindSummary, "GlowBoxTemplate")
    frame.addBindingHelpBox:SetWidth(300)
    frame.addBindingHelpBox:SetHeight(75)
    frame.addBindingHelpBox:ClearAllPoints()
    frame.addBindingHelpBox:SetPoint("TOPLEFT", frame, "TOPRIGHT", 10, -5)
    frame.addBindingHelpBox.Text = frame.addBindingHelpBox:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
    frame.addBindingHelpBox.Text:SetJustifyH("LEFT")
    frame.addBindingHelpBox.Text:SetJustifyV("MIDDLE")
    frame.addBindingHelpBox.Text:SetPoint("TOPLEFT", 10, -5)
    frame.addBindingHelpBox.Text:SetPoint("BOTTOMRIGHT", -10, 5)

    local addBindingHelpText = L["使用右侧的窗口选择法术或宏，然后使用 更改绑定 按钮设置单击或键绑定。可以使用下面的设置来配置绑定行为。"]
    frame.addBindingHelpBox.Text:SetText(addBindingHelpText)
    frame.addBindingHelpBox:SetFrameStrata("HIGH")
    frame.addBindingHelpBox:Show()

    frame.changeBinding:RegisterForClicks("AnyUp")
    page:ChangeBindingButton_Initialize(frame.changeBinding)

    frame.editMacro = CreateFrame("Button", nil, frame, "UIMenuButtonStretchTemplate")
    frame.editMacro:SetText(L["编辑宏"])
    frame.editMacro:ClearAllPoints()
    frame.editMacro:SetPoint("TOPLEFT", frame.changeBinding, "BOTTOMLEFT", 0, -5)
    frame.editMacro:SetWidth(125)

    frame.editMacro:SetScript("OnClick", function(button)
        local macrotext = page.draftBinding.macrotext
        local icon = page.draftBinding.icon

        config:SwitchToEditMacroPage(macrotext, icon)
    end)

    local bindConfigData = {
        -- [1]: Default bind-set
        {
            name = "Default",
            key = "default",
            label = L["在单位框体上激活 (|cffffd100default|r)"],
            tooltipTitle = L["Clique: 'default' 绑定集"],
            tooltip = L["属于 'default' 绑定集的绑定将始终在你的单位框体上处于活动状态，除非你用另一个绑定覆盖它。"],
        },
        -- [2]: OOC bind-set
        {
            name = "OOC",
            key = "ooc",
            label = L["仅在非战斗时施放 (|cffffd100ooc|r)"],
            tooltipTitle = L["Clique: 'ooc' 绑定集"],
            tooltip = L["属于 'ooc' 绑定集的绑定仅在玩家处于战斗状态时处于活动状态，无论此绑定属于哪个其他绑定集。一旦玩家进入战斗，这些绑定将不再有效，因此在为您经常使用的任何法术选择此绑定集时要小心。"],
        },
        -- [3]: Friend bind-set
        {
            name = "Friend",
            key = "friend",
            label = L["仅对友方单位施放 (|cffffd100friend|r)"],
            tooltipTitle = L["Clique: 'friend' 绑定集"],
            tooltip = L["属于 'friend' 绑定集的绑定仅在点击显示友方单位的单位展示框时激活，即那些你可以治疗和协助的单位。如果你点击了一个你无法治疗或协助的单位，什么都不会发生。"],
        },
        -- [4]: Enemy bind-set
        {
            name = "Enemy",
            key = "enemy",
            label = L["仅对敌方单位施放 (|cffffd100enemy|r)"],
            tooltipTitle = L["Clique: 'enemy' 绑定集"],
            tooltip = L["属于 'enemy' 绑定集的绑定在点击显示敌方单位（即可以攻击的单位）的单位帧时始终处于活动状态。如果你点击了一个你无法攻击的单位，什么都不会发生。"],
        },
        -- [5]: Hovercast bind-set
        {
            name = "Hovercast",
            key = "hovercast",
            label = L["仅当鼠标悬停在单位上时激活 (|cffffd100hovercast|r)"],
            tooltipTitle = L["Clique: 'hovercast' 绑定集"],
            tooltip = L["每当鼠标位于单位框体或游戏中的角色上时，属于 'hovercast' 绑定集的绑定就会处于活动状态。这允许您使用“悬浮施法”，即将鼠标悬停在游戏中的单位上并按下一个键对他们施法。这些绑定在单位框体上也处于活动状态。"],
        },
        -- [6]: Global bind-set
        {
            name = "Global",
            key = "global",
            label = L["始终处于活动状态，将覆盖游戏绑定 (|cffffd100global|r)"],
            tooltipTitle = L["Clique: 'global' 绑定集"],
            tooltip = L["属于 'global' 绑定集的绑定始终处于活动状态。如果法术需要目标，您将获得“施法手”，否则将施放法术。如果该法术是 AOE 法术，您将获得地面瞄准圈。"],
        },
    }

    for idx = 1, addon:GetNumTalentSpecs() do
        local name = "Spec" .. idx
        local key = name:lower()
        local specName = addon:GetTalentSpecName(idx)
        local label = L["天赋专精激活: %s (|cffffd100%s|r)"]:format(specName, key)
        table.insert(bindConfigData, {
            name = name,
            key = key,
            label = label,
            tooltipTitle = L["Clique: 此邦定集为 '%s'"]:format(specName),
            tooltip = L["属于此绑定集的绑定仅在玩家激活了给定的天赋专精时有效"],
        })
    end

    local bindSets = {}
    for idx, entry in ipairs(bindConfigData) do
        table.insert(bindSets, page:CreateBindSetCheckbox(frame, entry.name, entry.label, entry.tooltipTitle, entry.tooltip))
    end

    for idx, entry in ipairs(bindSets) do
        if idx == 1 then
            entry:ClearAllPoints()
            entry:SetPoint("TOPLEFT", frame.bindSummary, "BOTTOMLEFT", 5, -25)
        else
            entry:ClearAllPoints()
            entry:SetPoint("TOPLEFT", bindSets[idx-1], "BOTTOMLEFT", 0, -5)
        end
    end

    page.bindSetFrames = {}
    for idx, entry in ipairs(bindSets) do
        local key = entry.key:lower()
        page.bindSetFrames[key] = entry
    end
end

local function BindSetCheckbox_OnEnter(self, motion)
    if self.tooltip then
        config.ui.tooltip:SetOwner(self, "ANCHOR_RIGHT")
        if self.tooltipTitle then
            config.ui.tooltip:AddLine(self.tooltipTitle, 1, 1, 1)
        end
        config.ui.tooltip:AddLine(self.tooltip, nil, nil, nil, true)
        config.ui.tooltip:Show()
    end
end

local function BindSetCheckbox_OnLeave(self, motion)
    config.ui.tooltip:Hide()
end

local function BindSetCheckbox_OnClick(self)
    local checked = self:GetChecked()
    if checked then
        PlaySound(SOUNDKIT.IG_MAINMENU_OPTION_CHECKBOX_ON)
    else
        PlaySound(SOUNDKIT.IG_MAINMENU_OPTION_CHECKBOX_OFF)
    end

    page:ToggleBindSet(self.key, checked)
end

function page:CreateBindSetCheckbox(parent, key, text, tooltipTitle, tooltip)
    local nameKey = string.format("BindSetCheckButton%s", key)
    local button = CreateFrame("CheckButton", nil, parent)
    parent[nameKey] = button
    button.nameKey = nameKey
    button.key = key
    button.tooltipTitle = tooltipTitle
    button.tooltip = tooltip

    button:SetHeight(29)
    button:SetWidth(30)
    button:SetNormalAtlas("checkbox-minimal")
    button:SetPushedAtlas("checkbox-minimal")

    button.checkedTexture = button:CreateTexture(nil, "ARTWORK")
    button.checkedTexture:SetAllPoints()
    button.checkedTexture:SetAtlas("checkmark-minimal")
    button:SetCheckedTexture(button.checkedTexture)

    button.name = button:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
    button.name:SetJustifyH("LEFT")
    button.name:SetWordWrap(false)
    button.name:ClearAllPoints()
    button.name:SetPoint("LEFT", button, "RIGHT", 5, 0)
    button.name:SetText(text)

    button:SetScript("OnEnter", BindSetCheckbox_OnEnter)
    button:SetScript("OnLeave", BindSetCheckbox_OnLeave)
    button:SetScript("OnClick", BindSetCheckbox_OnClick)

    return button
end

local function ChangeBindingButton_OnEnter(self, motion)
    if motion and self.bindMode then
        self:EnableKeyboard(true)
    end
end

local function ChangeBindingButton_OnLeave(self, motion)
    self:EnableKeyboard(false)
end

local function ChangeBindingButton_CaptureAndEndBinding(key)
    local captured = addon:GetCapturedKey(key)
    if captured then
        page:ChangeBindingKey(captured)
        page.frame.changeBinding.bindMode = false
        page.frame.changeBindingArrow:Hide()
        page:UpdateEditPage()
    end
end

local function ChangeBindingButton_OnClick(self, button)
    if not self.bindMode then
        self.bindMode = true
        page.frame.changeBindingArrow:Show()
        self:EnableKeyboard(true)
    else
        ChangeBindingButton_CaptureAndEndBinding(button)
    end
end

local function ChangeBindingButton_OnKeyDown(self, key)
    if self.bindMode then
        ChangeBindingButton_CaptureAndEndBinding(key)
    end
end

local function ChangeBindingButton_OnMouseWheel(self, delta)
    if self.bindMode then
        local button = (delta > 0) and "MOUSEWHEELUP" or "MOUSEWHEELDOWN"
        ChangeBindingButton_CaptureAndEndBinding(button)
    end
end

function page:ChangeBindingButton_Initialize(button)
    button:SetScript("OnEnter", ChangeBindingButton_OnEnter)
    button:SetScript("OnLeave", ChangeBindingButton_OnLeave)
    button:SetScript("OnClick", ChangeBindingButton_OnClick)
    button:SetScript("OnKeyDown", ChangeBindingButton_OnKeyDown)
    button:SetScript("OnMouseWheel", ChangeBindingButton_OnMouseWheel)
    button:EnableKeyboard(false)
end

local pageMode = {
    EDIT = "edit",
    NEW = "new",
}

function page:ShowEditPageNewBinding()
    page:ResetPage()

    local draft = config:GetDefaultBindTable()
    page:SetDraftFromTable(draft)
    page.selectedBinding = nil
    page.mode = pageMode.NEW

    page:UpdateEditPage()
    page:Show()
end

function page:ShowEditPageSelectedBinding(bind)
    page:ResetPage()

    page:SetDraftFromTable(bind)
    page.selectedBinding = bind
    page.mode = pageMode.EDIT

    page:UpdateEditPage()
    page:Show()
end

function page:ResetPage()
    -- Reset to default behaviour

    page.selectedBinding = nil
    page.draftBinding = nil
    page.mode = nil

    page.rankRemoved = nil

    -- Visual display for change binding button and binding mode
    page.frame.changeBinding.bindMode = false
    page.frame.changeBindingArrow:Hide()
end

-- All edits happen on the 'draft' binding which then gets moved over
-- when things are saved, this is where we set the initial draft binding
function page:SetDraftFromTable(bind)
    page.draftBinding = {}

    local draft = page.draftBinding
    -- Action attributes
    draft.type  = bind.type
    draft.spell = bind.spell
    draft.spellSubName = bind.spellSubName
    draft.macro = bind.macro
    draft.macrotext = bind.macrotext
    -- Icon, key, unit
    draft.icon = bind.icon
    draft.key = bind.key
    draft.unit = bind.unit
    -- Need to copy sets over
    draft.sets = {}
    for k,v in pairs(bind.sets) do
        draft.sets[k] = v
    end
end

local function tableIsEmpty(tbl)
    for k, v in pairs(tbl) do
        return false
    end
    return true
end

-- Detect if the bind sets have changed between orig and draft
local function bindSetsChanged(orig, draft)
    local osets = orig.sets
    local dsets = draft.sets

    for k, v in pairs(osets) do
        if dsets[k] ~= v then
            return true
        end
    end

    for k, v in pairs(dsets) do
        if osets[k] ~= v then
            return true
        end
    end

    return false
end

-- Detect if the action has changed from orig to draft
local function actionChanged(orig, draft, rankRemoved)
    if orig.type ~= draft.type then
        return true
    elseif orig.spell ~= draft.spell then
        return true
    elseif orig.macro ~= draft.macro then
        return true
    elseif orig.macrotext ~= draft.macrotext then
        return true
    elseif orig.unit ~= draft.unit then
        return true
    elseif rankRemoved and orig.spellSubName ~= nil then
        -- If the original has a rank, but the new had rank removed
        return true
    end

    return false
end

-- Detect if the key binding has changed from orig to draft
local function keyChanged(orig, draft)
    return orig.key ~= draft.key
end

-- Detect if the icon has changed
local function iconChanged(orig, draft)
    return orig.icon ~= draft.icon
end


-- Actually update all of the elements of the page
function page:UpdateEditPage()
    -- Default values for a new or binding with missing values
    local name = L["新建绑定"]
    local icon = "Interface\\Icons\\INV_Misc_QuestionMark"
    local text = L["不属于任何绑定集"]
    local bindingText = L["未设置绑定"]

    local draft = page.draftBinding

    if draft.type then
        name = addon:GetBindingActionText(draft.type, draft, page.rankRemoved)
        icon = addon:GetBindingIcon(draft)
    end

    if draft.sets and not tableIsEmpty(draft.sets) then
        text = addon:GetBindingInfoText(draft)
    end

    if draft.key then
        bindingText = addon:GetBindingKeyComboText(draft)
    end

    -- Now check and see if those differ from the original
    local dirty = false

    local orig
    if page.mode == pageMode.EDIT then
        orig = page.selectedBinding
    elseif page.mode == pageMode.NEW then
        orig = config:GetDefaultBindTable()
    end

    if actionChanged(orig, draft, page.rankRemoved) then
        dirty = true
        name = string.format(L["|cff22ff22改变|r: %s"], name)
    end

    if bindSetsChanged(orig, draft) then
        dirty = true
        text = string.format(L["|cff22ff22改变|r: %s"], text)
    end

    if keyChanged(orig, draft) then
        dirty = true
        bindingText = string.format(L["|cff22ff22改变|r: %s"], bindingText)
    end

    if iconChanged(orig, draft) then
        dirty = true
    end

    local bindSummary = page.frame.bindSummary
    bindSummary.Name:SetText(name)
    bindSummary.Icon:SetTexture(icon)
    bindSummary.Text:SetText(text)
    bindSummary.BindingText:SetText(bindingText)

    -- Now update the bind-set checkboxes to reflect what is in draft
    for key, frame in pairs(page.bindSetFrames) do
        local checked = not not draft.sets[key]
        frame:SetChecked(checked)
    end

    -- If the binding is a spell and we have a rank
    if draft.type == "spell" and draft.spellSubName then
        page.frame.RemoveRankButton:Show()
    end

    -- Every "valid" binding needs an action and a key
    local valid = (draft.type ~= nil) and (draft.key ~= nil)

    if dirty and valid then
        page.frame.SaveButton:Enable()
    else
        page.frame.SaveButton:Disable()
    end

    if draft.type == "macro" and not draft.macro then
        -- This is a custom macro binding, so enable the edit button
        page.frame.editMacro:Show()
    else
        page.frame.editMacro:Hide()
    end
end


-- Change the binding action for the current edit page to the new action
-- removing the previous attributes if set
function page:ChangeBindingAction(entryType, entryId)
    local draft = page.draftBinding

    -- Need to remove the current attributes
    config:RemoveActionFromBinding(draft)

    -- Get the action attributes from the catalog entry and copy to draft
    local actionAttributes = config:GetActionAttributes(entryType, entryId)
    config:CopyActionFromTo(actionAttributes, draft)

    if actionAttributes.icon then
        draft.icon = actionAttributes.icon
    end

    page:UpdateEditPage()
end

-- Change the key binding for the current edit page
function page:ChangeBindingKey(key)
    local draft = page.draftBinding
    draft.key = key

    page:UpdateEditPage()
end

function page:SetMacrotextIcon(macrotext, icon)
    local draft = page.draftBinding

    draft.macrotext = macrotext
    draft.icon = icon

    page:UpdateEditPage()
end

function page:RemoveRankButton_OnClick(self, button)
    page.rankRemoved = true
    page:UpdateEditPage()
    self:Hide()
end

function page:ToggleBindSet(key, checked)
    local draft = page.draftBinding
    key = key:lower()

    checked = not not checked
    if checked then
        draft.sets[key] = true
    else
        draft.sets[key] = nil
    end
    page:UpdateEditPage()
end

function page:SaveButton_OnClick(self, button)
    local draft = page.draftBinding

    if page.mode == pageMode.NEW then
        -- Simple case of a new binding, draft should contain the new bind info
        addon:AddBinding(draft)
    else
        -- We are editing an existing binding
        local orig = page.selectedBinding
        if actionChanged(orig, draft, page.rankRemoved) then
            -- Remove previous action information from orig
            config:RemoveActionFromBinding(orig)

            -- Copy the action from draft to orig
            config:CopyActionFromTo(draft, orig)

            if page.rankRemoved then
                orig.spellSubName = nil
            end
        end

        if bindSetsChanged(orig, draft) then
            orig.sets = {}
            for key, value in pairs(draft.sets) do
                orig.sets[key] = value
            end
        end

        if keyChanged(orig, draft) then
            orig.key = draft.key
        end

        if iconChanged(orig, draft) then
            orig.icon = draft.icon
        end
    end

    -- Notify the addon that bindings have changed
    addon:FireMessage("BINDINGS_CHANGED")
    -- Clear the edit page to prevent consistency issues
    page:ResetPage()

    -- Swap back to the browse page
    config:SwitchToBrowsePage()
end

function page:CancelButton_OnClick(self, button)
    page:ResetPage()
    config:SwitchToBrowsePage()
end
